<?php
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../core/admin_guard.php'; // Admin Guard

// Check Admin Logic moved to admin_guard.php
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Admin - AutoTrade.AI</title>
    <style>
        body {
            margin: 0;
            font-family: 'Inter', sans-serif;
            background: #0b0e11;
            color: #d1d4dc;
        }

        .admin-header {
            background: #1b2029;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid #2b313f;
        }

        .logo {
            font-weight: 800;
            font-size: 1.2rem;
            color: #fff;
        }

        .logo span {
            color: #3b82f6;
        }

        .container {
            padding: 30px;
            max-width: 1000px;
            margin: 0 auto;
        }

        .card {
            background: #1b2029;
            border: 1px solid #2b313f;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.2);
        }

        h2 {
            margin-top: 0;
            color: #fff;
            font-size: 1.4rem;
            border-bottom: 1px solid #2b313f;
            padding-bottom: 15px;
            margin-bottom: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        th,
        td {
            text-align: left;
            padding: 12px;
            border-bottom: 1px solid #2b313f;
            font-size: 0.9rem;
        }

        th {
            color: #7e879a;
            font-weight: 600;
            font-size: 0.8rem;
            text-transform: uppercase;
        }

        .badge {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .badge-admin {
            background: rgba(59, 130, 246, 0.2);
            color: #3b82f6;
        }

        .badge-user {
            background: rgba(110, 118, 135, 0.2);
            color: #9ca3af;
        }

        .badge-premium {
            background: rgba(14, 203, 129, 0.2);
            color: #0ecb81;
        }

        .badge-free {
            background: rgba(238, 187, 0, 0.2);
            color: #eebb00;
        }

        .badge-active {
            background: rgba(14, 203, 129, 0.1);
            color: #0ecb81;
        }

        .badge-blocked {
            background: rgba(246, 70, 93, 0.1);
            color: #f6465d;
        }

        .btn {
            padding: 6px 12px;
            border-radius: 4px;
            font-size: 0.8rem;
            cursor: pointer;
            border: none;
            color: #fff;
            font-weight: 500;
        }

        .btn-blue {
            background: #3b82f6;
        }

        .btn-red {
            background: #f6465d;
        }

        .btn-green {
            background: #0ecb81;
        }

        .btn-outline {
            background: transparent;
            border: 1px solid #3b82f6;
            color: #3b82f6;
        }

        .btn:hover {
            opacity: 0.9;
        }

        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            z-index: 100;
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background: #1b2029;
            padding: 25px;
            border-radius: 8px;
            width: 100%;
            max-width: 400px;
            border: 1px solid #2b313f;
        }

        .form-group {
            margin-bottom: 15px;
        }

        label {
            display: block;
            margin-bottom: 5px;
            color: #aaa;
            font-size: 0.85rem;
        }

        input,
        select {
            width: 100%;
            padding: 10px;
            background: #0b0e11;
            border: 1px solid #2b313f;
            color: #fff;
            border-radius: 4px;
            box-sizing: border-box;
        }
    </style>
</head>

<body>
    <header class="admin-header">
        <div class="logo">AUTO<span>TRADE</span>.AI <span style="font-size:0.8rem; color:#555; margin-left:10px;">ADMIN
                PANEL</span></div>
        <div style="display:flex; gap:10px;">
            <a href="../index.php" class="btn btn-blue" style="text-decoration:none;">Go into Trade View ➔</a>
            <a href="../api/trade.php?action=logout" class="btn btn-red" style="text-decoration:none;">Logout</a>
        </div>
    </header>

    <div class="container">
        <div class="card">
            <h2>
                User Management
                <button class="btn btn-green" onclick="openAddUser()">+ Add User</button>
            </h2>
            <table id="userTable">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Role</th>
                        <th>Plan</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <!-- Populated by JS -->
                </tbody>
            </table>
        </div>

        <div class="card" style="margin-top: 30px;">
            <h2>
                AI Learning Memory
                <div style="display:flex; gap:10px;">
                    <button class="btn btn-blue" onclick="checkOutcomes()">🤖 Auto-Grade</button>
                    <button class="btn btn-green" onclick="triggerAI()">🧠 Deep Learn</button>
                    <button class="btn btn-outline" onclick="loadHistory()">↻ Refresh</button>
                </div>
            </h2>

            <!-- LEVEL 5: BRAIN STATUS -->
            <div style="margin-bottom:20px; background:#111; padding:15px; border-radius:6px; border:1px solid #333;">
                <h3 style="margin-top:0; font-size:1rem; color:#fff; display:flex; justify-content:space-between;">
                    🧠 Level 5 Brain Status (Strategy Weights)
                    <span style="font-size:0.8rem; color:#0ecb81;">Active & Learning</span>
                </h3>
                <div style="display:grid; grid-template-columns: repeat(5, 1fr); gap:10px;">
                    <?php
                    $w_file = __DIR__ . '/../strategy_weights.json';
                    $weights = file_exists($w_file) ? json_decode(file_get_contents($w_file), true) : [
                        "ema_trend" => 1.0,
                        "rsi_signal" => 1.0,
                        "volume_spike" => 1.0,
                        "news_bias" => 1.0,
                        "volatility_filter" => 1.0
                    ];
                    foreach ($weights as $k => $v) {
                        $color = ($v > 1.0) ? '#0ecb81' : (($v < 1.0) ? '#f6465d' : '#888');
                        echo "<div style='text-align:center; background:#1b2029; padding:8px; border-radius:4px;'>
                                <div style='font-size:0.7rem; color:#aaa; text-transform:uppercase;'>" . str_replace('_', ' ', $k) . "</div>
                                <div style='font-size:1.1rem; font-weight:bold; color:$color;'>" . number_format($v, 2) . "x</div>
                              </div>";
                    }
                    ?>
                </div>
            </div>

            <!-- Win Rate Stats -->
            <div
                style="display:flex; gap:20px; margin-bottom:15px; background:#111; padding:15px; border-radius:6px; border:1px solid #333;">
                <div style="text-align:center;">
                    <div style="font-size:0.8rem; color:#888;">Total Trades</div>
                    <div id="statTotal" style="font-size:1.5rem; font-weight:bold; color:#fff;">0</div>
                </div>
                <div style="text-align:center;">
                    <div style="font-size:0.8rem; color:#888;">Win Rate</div>
                    <div id="statWinRate" style="font-size:1.5rem; font-weight:bold; color:#0ecb81;">0%</div>
                </div>
                <div style="text-align:center;">
                    <div style="font-size:0.8rem; color:#888;">Accuracy Grade</div>
                    <div id="statGrade" style="font-size:1.5rem; font-weight:bold; color:#eebb00;">-</div>
                </div>
            </div>

            <table id="historyTable">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Symbol</th>
                        <th>Signal</th>
                        <th>Result (PnL)</th>
                        <th>AI Verify</th>
                    </tr>
                </thead>
                <tbody>
                    <!-- Populated by JS -->
                </tbody>
            </table>
        </div>
    </div>

    <!-- Add/Edit User Modal -->
    <div id="userModal" class="modal">
        <div class="modal-content">
            <h3 style="margin-top:0; color:#fff;">User Details</h3>
            <form id="userForm">
                <input type="hidden" id="editUserId">
                <div class="form-group">
                    <label>Username</label>
                    <input type="text" id="mUsername" required>
                </div>
                <div class="form-group">
                    <label>Password (Leave blank to keep current)</label>
                    <input type="text" id="mPassword">
                </div>
                <div class="form-group">
                    <label>Role</label>
                    <select id="mRole">
                        <option value="user">User</option>
                        <option value="admin">Admin</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Plan</label>
                    <select id="mPlan">
                        <option value="free">Free</option>
                        <option value="premium">Premium</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Status</label>
                    <select id="mStatus">
                        <option value="active">Active</option>
                        <option value="blocked">Blocked</option>
                    </select>
                </div>
                <div style="display:flex; gap:10px; justify-content:flex-end; margin-top:20px;">
                    <button type="button" class="btn btn-outline" onclick="closeModal()">Cancel</button>
                    <button type="submit" class="btn btn-blue">Save</button>
                </div>
            </form>
        </div>
    </div>

    <script>
        async function loadUsers() {
            const res = await fetch('../api/trade.php?action=get_users');
            const users = await res.json();
            const tbody = document.querySelector('#userTable tbody');
            tbody.innerHTML = '';

            users.forEach(u => {
                const tr = document.createElement('tr');
                tr.innerHTML = `
                    <td>#${u.id}</td>
                    <td>${u.username}</td>
                    <td><span class="badge badge-${u.role}">${u.role.toUpperCase()}</span></td>
                    <td><span class="badge badge-${u.plan}">${u.plan.toUpperCase()}</span></td>
                    <td><span class="badge badge-${u.status === 'active' ? 'active' : 'blocked'}">${u.status.toUpperCase()}</span></td>
                    <td>
                        <button class="btn btn-outline" style="padding:4px 8px; font-size:0.7rem;" onclick='openEdit(${JSON.stringify(u)})'>Edit</button>
                    </td>
                `;
                tbody.appendChild(tr);
            });
        }

        function openAddUser() {
            document.getElementById('editUserId').value = '';
            document.getElementById('userForm').reset();
            document.getElementById('userModal').style.display = 'flex';
        }

        function openEdit(u) {
            document.getElementById('editUserId').value = u.id;
            document.getElementById('mUsername').value = u.username;
            document.getElementById('mPassword').value = '';
            document.getElementById('mRole').value = u.role;
            document.getElementById('mPlan').value = u.plan;
            document.getElementById('mStatus').value = u.status;
            document.getElementById('userModal').style.display = 'flex';
        }

        function closeModal() {
            document.getElementById('userModal').style.display = 'none';
        }

        document.getElementById('userForm').addEventListener('submit', async function (e) {
            e.preventDefault();
            const id = document.getElementById('editUserId').value;
            const u = document.getElementById('mUsername').value;
            const p = document.getElementById('mPassword').value;
            const r = document.getElementById('mRole').value;
            const pl = document.getElementById('mPlan').value;
            const s = document.getElementById('mStatus').value;

            const action = id ? 'edit_user' : 'add_user';
            const body = `id=${id}&username=${u}&password=${p}&role=${r}&plan=${pl}&status=${s}`;

            const res = await fetch('../api/trade.php?action=' + action, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: body
            });
            const data = await res.json();
            if (data.success) {
                closeModal();
                loadUsers();
            } else {
                alert('Error: ' + data.error);
            }
        });

        loadUsers();
        loadHistory();

        async function loadHistory() {
            // Add timestamp to prevent caching
            const res = await fetch('../api/trade.php?action=get_trade_history&t=' + new Date().getTime());
            const logs = await res.json();
            const tbody = document.querySelector('#historyTable tbody');
            tbody.innerHTML = '';

            // Stats Calculation
            let wins = 0;
            let completed = 0;

            // Show latest first
            [...logs].reverse().forEach(log => {
                let statusColor = 'gray';
                if (log.status === 'WIN') { wins++; completed++; statusColor = 'success'; }
                if (log.status === 'LOSS') { completed++; statusColor = 'danger'; }
                if (log.status === 'OPEN') statusColor = 'warning';

                const tr = document.createElement('tr');
                tr.innerHTML = `
                    <td>${new Date(log.timestamp * 1000).toLocaleString()}</td>
                    <td style="font-weight:700;">${log.symbol}</td>
                    <td>${log.signal}</td>
                    <td style="color:${log.pnl > 0 ? '#0ecb81' : (log.pnl < 0 ? '#f6465d' : '#888')}">${log.pnl > 0 ? '+' : ''}${log.pnl}%</td>
                    <td><span class="badge badge-${statusColor}">${log.status}</span></td>
                `;
                tbody.appendChild(tr);
            });

            // Update Stats Display
            const totalEl = document.getElementById('statTotal');
            const winRateEl = document.getElementById('statWinRate');
            const gradeEl = document.getElementById('statGrade');

            if (totalEl) totalEl.innerText = completed;

            let rate = 0;
            if (completed > 0) rate = Math.round((wins / completed) * 100);

            if (winRateEl) {
                winRateEl.innerText = rate + "%";
                winRateEl.style.color = rate >= 50 ? '#0ecb81' : '#f6465d';
            }

            if (gradeEl) {
                let grade = '-';
                let color = '#888';
                if (completed > 0) {
                    if (rate >= 80) { grade = 'A+'; color = '#0ecb81'; }
                    else if (rate >= 65) { grade = 'B'; color = '#3b82f6'; }
                    else if (rate >= 50) { grade = 'C'; color = '#eebb00'; }
                    else { grade = 'D'; color = '#f6465d'; }
                }
                gradeEl.innerText = grade;
                gradeEl.style.color = color;
            }
        }

        async function checkOutcomes() {
            const btn = document.querySelector('button[onclick="checkOutcomes()"]');
            if (btn) {
                btn.innerText = "Checking...";
                btn.disabled = true;
            }

            try {
                const res = await fetch('../api/trade.php?action=assess_outcomes');
                const text = await res.text(); // Get raw text
                console.log(text); // Debug

                try {
                    const data = JSON.parse(text);
                    if (data.success) {
                        alert(`Check Complete! Updated ${data.updated} trades.`);
                        loadHistory();
                    } else {
                        alert("Error: " + (data.msg || "Unknown"));
                    }
                } catch (e) {
                    // This catches PHP Fatal Errors
                    alert("Server Error: " + text.substring(0, 100));
                }

            } catch (err) {
                console.error(err);
                alert("Network Request Failed");
            }

            if (btn) {
                btn.innerText = "🤖 Auto-Grade Outcomes";
                btn.disabled = false;
            }
        }

        async function triggerAI() {
            const btn = document.querySelector('button[onclick="triggerAI()"]');
            btn.innerText = "Learning...";
            btn.disabled = true;
            const res = await fetch('../api/trade.php?action=trigger_learning');
            const data = await res.json();
            if (data.status === 'Success') {
                alert(`Deep Learning Complete!\nAssessed: ${data.learned_from} trades.\nStrategy Weights Updated.`);
                location.reload(); // Refresh to show new brain weights
            } else {
                alert("Error: " + data.status);
            }
            btn.innerText = "🧠 Deep Learn";
            btn.disabled = false;
        }
    </script>
</body>

</html>