<?php
// ai/market_state.php
// LEVEL 1: MARKET STATE AWARENESS BRAIN
// Determines if market is TRENDING, SIDEWAYS, or VOLATILE

if (!function_exists('detect_market_state')) {
    function detect_market_state($candles)
    {
        $count = count($candles);
        if ($count < 20) {
            return [
                "state" => "UNKNOWN",
                "confidence" => 0,
                "reason" => "Insufficient Data"
            ];
        }

        // 1. Extract Data
        $closes = array_column($candles, 'c');
        $highs = array_column($candles, 'h');
        $lows = array_column($candles, 'l');
        $volumes = array_column($candles, 'v');

        // 2. Calculate Indicators
        $adx = calculate_adx_dynamic($candles, 14); // Need robust ADX func
        $atr = calculate_atr_dynamic($candles, 14);

        // 3. Volatility Analysis (ATR Ratio)
        // Compare current ATR to long-term average ATR (if we had history, but here we use recent deviation)
        // Simple Volatility Check: (High - Low) / Close
        $recent_volatility = [];
        for ($i = 0; $i < 5; $i++) {
            $idx = $count - 1 - $i;
            $range = $highs[$idx] - $lows[$idx];
            $recent_volatility[] = ($range / $closes[$idx]) * 100;
        }
        $avg_vol_pct = array_sum($recent_volatility) / 5;

        // 4. Volume Analysis
        $avg_vol = array_sum(array_slice($volumes, -20)) / 20;
        $curr_vol = end($volumes);
        $vol_ratio = ($avg_vol > 0) ? ($curr_vol / $avg_vol) : 1;

        // 5. DECISION LOGIC
        $state = "SIDEWAYS"; // Default
        $conf = 0.5;
        $reason = "Normal Activity";

        // RULE 1: HIGH VOLATILITY (Panic/News)
        if ($avg_vol_pct > 0.5) { // If candles move > 0.5% on average (High for low timeframe)
            $state = "VOLATILE";
            $conf = 0.9;
            $reason = "High Candle Range (>0.5%)";
        }
        // RULE 2: TRENDING (ADX)
        else if ($adx > 25) {
            $state = "TRENDING";
            // Check direction
            $last_c = end($closes);
            $ema20 = calculate_ema_dynamic($closes, 20);
            $direction = ($last_c > $ema20) ? "UP" : "DOWN";

            $conf = min(0.95, 0.5 + ($adx / 100)); // ADX 25->0.75, ADX 50->1.0
            $reason = "Strong Trend (ADX $adx) - $direction";
        }
        // RULE 3: SIDEWAYS (Low ADX)
        else {
            $state = "SIDEWAYS";
            $conf = 0.8; // High confidence it is sideways
            $reason = "Weak Trend (ADX $adx)";

            if ($vol_ratio < 0.5) {
                $reason .= " & Low Volume";
            }
        }

        return [
            "state" => $state,
            "confidence" => round($conf, 2),
            "adx" => $adx,
            "vol_ratio" => round($vol_ratio, 2),
            "reason" => $reason
        ];
    }
}

// --- HELPER FUNCTIONS FOR AI BRAIN (Isolated to ensure no conflict) ---

function calculate_ema_dynamic($data, $period)
{
    if (count($data) <= $period)
        return 0;
    $k = 2 / ($period + 1);
    $ema = $data[0];
    foreach ($data as $price) {
        $ema = ($price * $k) + ($ema * (1 - $k));
    }
    return $ema;
}

function calculate_atr_dynamic($candles, $period)
{
    // True Range logic
    $tr_sum = 0;
    $count = count($candles);
    if ($count < $period + 1)
        return 0;

    // Simple average of TR for speed in PHP
    $trs = [];
    for ($i = 1; $i < $count; $i++) {
        $h = $candles[$i]['h'];
        $l = $candles[$i]['l'];
        $pc = $candles[$i - 1]['c'];

        $tr = max($h - $l, abs($h - $pc), abs($l - $pc));
        $trs[] = $tr;
    }

    $atr_slice = array_slice($trs, -$period);
    return array_sum($atr_slice) / count($atr_slice);
}

function calculate_adx_dynamic($candles, $period)
{
    // Simplified ADX approximation for PHP Speed
    // Full ADX requires recursive smoothing which needs history. 
    // We will use a standard component approach suitable for 100 candles.

    $count = count($candles);
    if ($count < $period * 2)
        return 20; // Default neutral

    $tr = [];
    $dm_plus = [];
    $dm_minus = [];

    for ($i = 1; $i < $count; $i++) {
        $h = $candles[$i]['h'];
        $l = $candles[$i]['l'];
        $ph = $candles[$i - 1]['h'];
        $pl = $candles[$i - 1]['l'];
        $pc = $candles[$i - 1]['c'];

        $tr[] = max($h - $l, abs($h - $pc), abs($l - $pc));

        $up = $h - $ph;
        $down = $pl - $l;

        $dm_plus[] = ($up > $down && $up > 0) ? $up : 0;
        $dm_minus[] = ($down > $up && $down > 0) ? $down : 0;
    }

    // Smooth (Wilder's Smoothing usually, simple SMA here for stability on small data)
    // ADX is sensitive, let's use standard averaging
    $tr_avg = array_avg_last($tr, $period);
    $dm_plus_avg = array_avg_last($dm_plus, $period);
    $dm_minus_avg = array_avg_last($dm_minus, $period);

    if ($tr_avg == 0)
        return 0;

    $di_plus = ($dm_plus_avg / $tr_avg) * 100;
    $di_minus = ($dm_minus_avg / $tr_avg) * 100;

    $dx = 0;
    if (($di_plus + $di_minus) > 0) {
        $dx = abs($di_plus - $di_minus) / ($di_plus + $di_minus) * 100;
    }

    // Return DX as ADX proxy for snapshot (Real ADX is smoothed DX)
    // For "Current State", recent DX is actually more responsive than smoothed ADX
    return round($dx, 2);
}

function array_avg_last($arr, $n)
{
    $slice = array_slice($arr, -$n);
    if (count($slice) == 0)
        return 0;
    return array_sum($slice) / count($slice);
}
?>