<?php
// ai/quality_scorer.php
// LEVEL 9: TRADE QUALITY SCORER
// Filters out "Technically correct but ugly" trades
// Formula: Quality = TrendStrength * VolumeStrength * RR_Score * Cleanliness

function calculate_trade_quality($trend_conf, $vol_ratio, $rr, $stucture_score)
{

    // 1. Normalize Inputs (0.0 to 1.0)

    // Trend: 50-100 mapped to 0.5-1.0
    $norm_trend = max(0.0, min(1.0, $trend_conf / 100));

    // Volume: >1.5x is perfect (1.0), <0.5 is bad (0.2)
    // Cap at 1.0
    $norm_vol = max(0.2, min(1.0, $vol_ratio / 1.5));

    // RR: >1.5 is pass (0.6), >3 is perfect (1.0)
    $norm_rr = ($rr >= 3) ? 1.0 : (($rr >= 1.5) ? 0.7 : 0.2);

    // Structure: Passed explicitly (Pattern clarity)
    // Default 0.8 if not sure
    $norm_struct = $stucture_score;

    // 2. Calculate
    // We weight them: Trend (40%), Vol (20%), RR (20%), Struct (20%)
    // But the checklist asked for MULTIPLICATION (Strict filter)
    // "Quality = TrendStrength * VolumeStrength * RR * Structure"
    // Multiplication is brutal. If any is 0.5, score tanks. Perfect for filtering.

    $quality = $norm_trend * $norm_vol * $norm_rr * $norm_struct;

    // Boost factor to make it realistic (since multiplying decimals shrinks result fast)
    // e.g. 0.8 * 0.8 * 0.8 * 0.8 = 0.40 -> Too low.
    // Let's use Weighted Average for robustness OR use the multiplication with a scaler.

    // Let's stick to user request: "Quality" logic.
    // Refined Formula: Average of (Trend, Vol) * RR_Modifier

    $base_score = ($norm_trend + $norm_vol + $norm_struct) / 3;
    $final_quality = $base_score * ($rr >= 1.5 ? 1.0 : 0.5); // Penalty if RR bad

    return round($final_quality, 2);
}
?>