/* app.js v14.5 - Disciplined AI Assistant UI */
console.log("AutoTrade AI Frontend Loaded: v14.5 (Disciplined Polish)");

const THEME = {
    bg: '#131722',
    grid: '#2b313f',
    text: '#d1d4dc',
    up: '#0ecb81',
    down: '#f6465d',
    crosshair: '#ffffff',
};

const API_TOKEN = '7f8a9d1e2b3c4f5a6b7c8d9e0f1a2b3c4d5e6f7a8b9c0d1e2f3a4b5c6d7e8f9';
const API_HEADERS = {
    'Content-Type': 'application/json',
    'X-API-TOKEN': API_TOKEN
};

const state = {
    symbol: 'APOLLOHOSP.NS',
    interval: '5m', // Standardized to 5m for Professional Analysis consistency
    market: 'IN', // 'IN' or 'US'
    data: [],
    offset: 0,
    zoom: 150,
    vZoom: 1.0,
    drag: { active: false, x: 0, y: 0 },
    mouse: { x: 0, y: 0 }
};

function getCurrency() {
    return state.market === 'US' ? '$' : '₹';
}

function switchMarket(market) {
    console.log("Switching Market to:", market);
    state.market = market;

    // Update Stock List
    if (market === 'US') {
        STOCK_LIST = STOCK_LIST_US;
        state.symbol = 'AAPL'; // Default US Stock
    } else {
        STOCK_LIST = STOCK_LIST_IN;
        state.symbol = 'APOLLOHOSP.NS'; // Default IN Stock
    }

    // Refresh Sidebar List
    const wh = document.getElementById('watchlistHeader');
    if (wh) wh.innerText = (market === 'US') ? 'US MARKET GIANTS' : 'NIFTY 50 WATCHLIST';

    const ul = document.getElementById('stockListUl');
    if (ul) {
        ul.innerHTML = ''; // Clear
        STOCK_LIST.forEach(s => {
            const li = document.createElement('li');
            li.className = 'stock-item';
            li.innerText = s.replace('.NS', ''); // Clean name
            li.onclick = () => loadStock(s, li);
            if (s === state.symbol) li.classList.add('active');
            ul.appendChild(li);
        });
    }

    // Trigger Data Fetch
    state.data = [];
    // Force UI Update
    loadStock(state.symbol, null);

    // Refresh Ticker Immediately
    if (typeof runScanner === 'function') {
        // slight delay to ensure state update
        setTimeout(runScanner, 100);
    }
}

const canvas = document.getElementById('candleChart');
const ctx = canvas.getContext('2d');
const ui = {
    ticker: document.getElementById('currentTicker'),
    price: document.getElementById('displayPrice'),
    change: document.getElementById('displayChange'),
    dataSource: document.getElementById('dataSource')
};

// Stocks are loaded from stocks.js (ALL_STOCKS_IN, ALL_STOCKS_US)
let STOCK_LIST_IN = [];
let STOCK_LIST_US = [];
if (typeof ALL_STOCKS_IN !== 'undefined') STOCK_LIST_IN = ALL_STOCKS_IN;
if (typeof ALL_STOCKS_US !== 'undefined') STOCK_LIST_US = ALL_STOCKS_US;


let STOCK_LIST = STOCK_LIST_IN; // Default to India

function init() {
    window.addEventListener('resize', resize);
    canvas.addEventListener('mousedown', onMouseDown);
    window.addEventListener('mouseup', onMouseUp);
    window.addEventListener('mousemove', onMouseMove);
    canvas.addEventListener('wheel', onWheel, { passive: false });

    // Auto-Refresh Market Data every 5 minutes (300,000 ms)
    setInterval(() => {
        console.log('🔄 AI Auto-Refresh Triggered (5-min interval)');
        fetchMarket();
    }, 300000);

    // Wire Header Search Input
    const input = document.getElementById('headerSearch');
    if (input) {
        input.addEventListener("keypress", function (event) {
            if (event.key === "Enter") {
                event.preventDefault();
                handleHeaderSearch();
                closeSuggestions();
            }
        });

        // Search Suggestions
        input.addEventListener("input", function () {
            const val = this.value.toUpperCase();
            if (val.length < 1) { closeSuggestions(); return; }
            const matches = STOCK_LIST.filter(s => s.includes(val) || s.replace('.NS', '').includes(val));
            showSuggestions(matches);
        });

        document.addEventListener('click', function (e) {
            if (e.target !== input) closeSuggestions();
        });
        // Wire Analyze Button (Header)
        const btnAnalyze = document.getElementById('btnAnalyze');
        if (btnAnalyze) btnAnalyze.onclick = () => { handleHeaderSearch(); };

        // Wire Settings Button
        const btnSettings = document.querySelector('.nav-item:nth-child(2)'); // Settings is 2nd item
        if (btnSettings) {
            btnSettings.onclick = () => {
                if (USER_INFO.role === 'admin') {
                    window.location.href = 'admin.php';
                } else {
                    openSettingsModal();
                }
            };
        }



        // Wire Dashboard Button
        const btnDash = document.querySelector('.nav-item:nth-child(1)');
        if (btnDash) {
            btnDash.onclick = () => {
                // Already on dashboard, maybe refresh or scroll top
            };
        }

        // Wire Mobile Menu Button (Hamburger)
        const btnMenu = document.getElementById('mobileMenuBtn');
        const sidebar = document.querySelector('.sidebar');
        const btnClose = document.getElementById('sidebarCloseBtn'); // New Close Btn

        if (btnMenu && sidebar) {
            btnMenu.onclick = () => {
                sidebar.classList.add('active'); // Open
            };

            // Close Button Logic
            if (btnClose) {
                btnClose.onclick = () => {
                    sidebar.classList.remove('active'); // Close
                }
            }

            document.addEventListener('click', (e) => {
                if (sidebar.classList.contains('active')) {
                    if (!sidebar.contains(e.target) && e.target !== btnMenu) {
                        sidebar.classList.remove('active');
                    }
                }
            });
        }

        // Populate Dynamic List
        renderStockList(STOCK_LIST.slice(0, 50));
    }

    renderStockList(STOCK_LIST.slice(0, 50));

    // Wire existing buttons
    const btnIn = document.getElementById('btnZoomIn');
    const btnOut = document.getElementById('btnZoomOut');
    if (btnIn) btnIn.onclick = () => { state.zoom = Math.max(10, state.zoom - 20); draw(); };
    if (btnOut) btnOut.onclick = () => { state.zoom = Math.min(1000, state.zoom + 20); draw(); };

    setInterval(fetchMarket, 5000);
    resize();
    fetchMarket();
    initGeneralNews();
}

// Show Suggestions Function
function showSuggestions(list) {
    const box = document.getElementById('searchSuggestions');
    if (!box) return;
    box.innerHTML = '';

    if (list.length === 0) {
        box.style.display = 'none';
        return;
    }

    list.slice(0, 10).forEach(sym => {
        const div = document.createElement('div');
        div.className = 'suggestion-item';
        div.innerText = sym;
        div.onclick = () => {
            const inp = document.getElementById('headerSearch');
            if (inp) inp.value = sym;
            handleHeaderSearch();
            closeSuggestions();
        };
        box.appendChild(div);
    });
    box.style.display = 'block';
}

function closeSuggestions() {
    const box = document.getElementById('searchSuggestions');
    if (box) box.style.display = 'none';
}

window.handleHeaderSearch = function () {
    const input = document.getElementById('headerSearch');
    let sym = input.value.toUpperCase().trim();
    if (!sym) return;

    if (state.market === 'IN' && !sym.includes('.') && !sym.includes('^')) {
        sym += '.NS';
    }

    // Update Input value back to clean Symbol
    input.value = sym;

    document.querySelectorAll('.stock-item').forEach(e => e.classList.remove('active'));

    state.symbol = sym;
    // Update Header Ticker immediately
    const headerTicker = document.getElementById('currentTicker');
    if (headerTicker) headerTicker.innerText = sym;

    state.data = [];
    state.offset = 0;
    fetchMarket();
}

function renderStockList(list) {
    const ul = document.getElementById('stockListUl');
    if (!ul) return;
    ul.innerHTML = '';

    list.forEach(s => {
        const li = document.createElement('li');
        li.className = 'stock-item' + (s === state.symbol ? ' active' : '');
        let displayName = s.replace('.NS', '');
        // Friendly Names for Indices
        if (s === '^NSEI') displayName = 'NIFTY 50';
        if (s === '^NSEBANK') displayName = 'BANK NIFTY';
        if (s === '^CNXIT') displayName = 'NIFTY IT';
        if (s === '^BSESN') displayName = 'SENSEX';
        if (s === '^GSPC') displayName = 'S&P 500';
        if (s === '^DJI') displayName = 'DOW JONES';
        if (s === '^IXIC') displayName = 'NASDAQ';

        li.innerText = displayName;
        li.onclick = () => loadStock(s, li);
        ul.appendChild(li);
    });
}


window.vZoomIn = function () { state.vZoom *= 1.1; draw(); };
window.vZoomOut = function () { state.vZoom *= 0.9; draw(); };
window.changeTimeframe = function (i, b) {
    state.interval = i;
    state.data = [];
    document.querySelectorAll('.time-btn').forEach(btn => btn.classList.remove('active'));
    if (b) b.classList.add('active');
    fetchMarket();
}
window.loadStock = function (s, el) {
    state.symbol = s;
    if (el) {
        document.querySelectorAll('.stock-item').forEach(e => e.classList.remove('active'));
        el.classList.add('active');
    }
    state.data = [];
    state.offset = 0;
    fetchMarket();
}

// --- PRO UPGRADE: AUDIO ALERTS UTILITY ---
const AudioAlerts = {
    buy: new Audio('https://assets.mixkit.co/active_storage/sfx/2869/2869-preview.mp3'), // Chime
    sell: new Audio('https://assets.mixkit.co/active_storage/sfx/995/995-preview.mp3'), // Alert
    lastSignal: null, // To prevent spamming

    play(signal) {
        if (this.lastSignal === signal) return; // Don't repeat for same cached signal
        this.lastSignal = signal;

        // Only play if user interaction happened (Browser Policy)
        // We assume user clicked something to start.
        if (signal === 'BUY') {
            this.buy.play().catch(e => console.log("Audio Blocked (Click page first)", e));
        } else if (signal === 'SELL') {
            this.sell.play().catch(e => console.log("Audio Blocked", e));
        }
    }
};

async function fetchMarket() {
    try {
        const res = await fetch(`api/trade.php?action=get_data&symbol=${state.symbol}&interval=${state.interval}&token=${API_TOKEN}&_t=${Date.now()}`, { headers: API_HEADERS });
        const data = await res.json();
        if (data && data.candles && data.candles.length > 0) {
            state.data = data.candles;
            ui.price.innerText = getCurrency() + data.price.toFixed(2);
            ui.change.innerText = data.change_pct.toFixed(2) + '%';
            ui.change.className = data.change_pct >= 0 ? 'price-change p-up' : 'price-change p-down';

            // --- UPDATE HEADER DETAILS ---
            const dSym = document.getElementById('detailSymbol');
            const dHigh = document.getElementById('detailHigh');
            const dLow = document.getElementById('detailLow');
            const dVol = document.getElementById('detailVol');

            if (dSym) dSym.innerText = state.symbol;

            // Calculate High/Low/Vol from loaded data (approximate for session)
            let high = -Infinity, low = Infinity, vol = 0;
            // Scan last 375 candles (approx 1 trading day of minutes) or all if daily
            const lookback = state.interval.includes('d') ? 1 : Math.min(data.candles.length, 375);
            const subset = data.candles.slice(-lookback);

            subset.forEach(c => {
                if (c.h > high) high = c.h;
                if (c.l < low) low = c.l;
                vol += c.v;
            });

            if (dHigh) dHigh.innerText = getCurrency() + high.toFixed(2);
            if (dLow) dLow.innerText = getCurrency() + low.toFixed(2);
            if (dVol) dVol.innerText = (vol / 1000000).toFixed(2) + 'M'; // Show in Millions

            // Sync Header Search Input if not focused
            const inp = document.getElementById('headerSearch');
            if (inp && document.activeElement !== inp) inp.value = state.symbol;
            ui.dataSource.innerText = 'Source: ' + data.source;

            // Safety Check (Zomato Fix)
            if (!data.analysis) {
                data.analysis = {
                    intraday: { signal: 'WAIT', conf: 0, text_ml: 'Data insufficient', text_en: 'Data insufficient' },
                    swing: { signal: 'WAIT', conf: 0 },
                    longterm: { signal: 'WAIT', conf: 0 }
                };
            }

            // PRICE RECOVERY
            if ((!data.price || data.price == 0) && data.candles && data.candles.length > 0) {
                data.price = data.candles[data.candles.length - 1].c;
                ui.price.innerText = getCurrency() + data.price.toFixed(2); // Re-update UI
            }

            updateSignals(data);
            requestAnimationFrame(draw);
        }
    } catch (e) { console.error(e); }
}

// --- AI TAB SWITCHING (Linked to Bottom Box) ---
window.switchAI = function (mode, btn) {
    // 1. Update Tabs
    document.querySelectorAll('.ai-tab').forEach(b => b.classList.remove('active'));
    if (btn) btn.classList.add('active');

    // 2. Show Content
    document.querySelectorAll('.ai-content').forEach(c => c.classList.remove('active'));
    document.getElementById('ai-content-' + mode).classList.add('active');

    // 3. Render Trade Plan for this mode (Dynamic Switch)
    renderTradePlan(mode);
};

// --- PROFESSIONAL V14.5 DISPLAY LOGIC ---
let CURRENT_ANALYSIS = null; // Store for switching

function updateSignals(data) {
    if (!data.analysis) return;
    const a = data.analysis;
    CURRENT_ANALYSIS = a; // Store globally

    // --- 1. CHECK LOCK (Strict Frontend Guard) ---
    // If locked, block EVERYTHING immediately
    if (a.is_locked || USER_INFO.plan !== 'premium') {
        const aiCard = document.querySelector('.ai-card');

        // Reset Needle to Center if blocked (Don't leak sentiment)
        const needle = document.getElementById('sentiment-needle');
        if (needle) needle.style.left = '50%';

        if (aiCard && !aiCard.querySelector('.premium-lock-overlay')) {
            const overlay = document.createElement('div');
            overlay.className = 'premium-lock-overlay';
            overlay.innerHTML = `
                <div class="lock-icon">🔒</div>
                <div class="lock-msg">Premium Analysis</div>
                <button class="btn-upgrade" onclick="openSettingsModal()">Upgrade</button>
            `;
            aiCard.appendChild(overlay);
        }
        return; // STOP UPDATING UI
    } else {
        // Remove overlay if exists (User upgraded or admin)
        const overlay = document.querySelector('.premium-lock-overlay');
        if (overlay) overlay.remove();
    }

    // Helper to update specific horizon
    const updateHorizon = (mode, sig) => {
        const elSig = document.getElementById('ai-signal-' + mode);
        const elConf = document.getElementById('ai-conf-' + mode);
        const elConfVal = document.getElementById('ai-conf-val-' + mode);
        const elDesc = document.getElementById('ai-desc-' + mode);
        const container = document.getElementById('ai-content-' + mode);

        // Language Select
        const langSelect = document.getElementById('langSelect');
        const lang = langSelect ? langSelect.value : 'ml';
        let textToShow = sig.text_ml;
        if (lang === 'en') textToShow = sig.text_en;
        if (lang === 'hi') textToShow = sig.text_hi;

        if (elSig) {
            elSig.innerText = sig.signal;
            elSig.style.color = sig.color === 'bullish' ? '#0ecb81' : (sig.color === 'bearish' ? '#f6465d' : '#eebb00');
        }
        if (elConf) elConf.style.width = sig.conf + '%';
        if (elConfVal) {
            // New Strength Label Logic
            let strengthText = "";
            if (sig.strength) strengthText = `<div style="font-size:0.6rem; color:#aaa; margin-top:2px; font-weight:normal;">Strength: <span style="color:#fff;">${sig.strength}</span></div>`;
            elConfVal.innerHTML = `<div style="font-size:1rem; font-weight:700;">${sig.conf}%</div>${strengthText}`;
        }

        if (elDesc) elDesc.innerText = textToShow;

        // **USER REQUEST: REMOVED Intraday Text Override**
        // News will be handled by independent rotation logic
        /**
         * Original Code Removed:
         * if (mode === 'intraday') { ... }
         */

        // --- 1. CONFIDENCE BREAKDOWN & PSYCHOLOGY (Inject below description) ---
        const existingBreakdown = container.querySelector('.pro-breakdown');
        if (existingBreakdown) existingBreakdown.remove();

        if (sig.signals && Object.keys(sig.signals).length > 0) {
            const div = document.createElement('div');
            div.className = 'pro-breakdown';
            div.style.marginTop = '10px';
            div.style.fontSize = '0.75rem';
            div.style.color = '#a0aab9';

            // Breakdown Title
            div.innerHTML = `<div style="border-bottom:1px solid #2b313f; margin-bottom:5px; padding-bottom:2px;">Confidence Logic:</div>`;

            // Breakdown List
            for (const k in sig.signals) {
                const s = sig.signals[k];
                const color = s.weight > 0 ? '#0ecb81' : (s.weight < 0 ? '#f6465d' : '#777');
                const sign = s.weight > 0 ? '+' : '';
                div.innerHTML += `
                    <div style="display:flex; justify-content:space-between; margin-bottom:2px;">
                        <span>• ${s.desc}</span>
                        <span style="color:${color}; font-weight:600;">${sign}${s.weight}</span>
                    </div>
                `;
            }

            // Expanded Actionable AI Note (Bullets)
            const noteColor = sig.conf > 80 ? '#0ecb81' : (sig.conf < 40 ? '#f6465d' : '#eebb00');
            // Default bullets if not provided by backend (or construct from data)
            let bullets = [];
            if (sig.conf > 75) {
                bullets = ["High probability setup", "Size normal", "Ride trend"];
            } else if (sig.conf < 40) {
                bullets = ["Low probability", "Avoid trading", "Wait for confirmed break"];
            } else {
                bullets = ["Moderate probability breakout", "Trade only with strict SL", "Avoid adding positions"];
            }

            div.innerHTML += `
                <div style="margin-top:10px; padding:8px; background:rgba(255,255,255,0.05); border-left:3px solid ${noteColor};">
                    <div style="font-weight:bold; color:${noteColor}; margin-bottom:4px;">AI Note:</div>
                    <ul style="margin:0; padding-left:15px; color:#ccc; list-style-type:disc; line-height:1.4;">
                        ${bullets.map(b => `<li>${b}</li>`).join('')}
                    </ul>
                </div>
            `;

            container.appendChild(div);
        }
    };

    // Update All Horizons (And check for Sure Shot)
    if (a.intraday) { updateHorizon('intraday', a.intraday); checkSureShot(a.intraday); }
    if (a.swing) { updateHorizon('swing', a.swing); checkSureShot(a.swing); }
    if (a.longterm) { updateHorizon('longterm', a.longterm); checkSureShot(a.longterm); }

    // Update Sentiment Meter
    if (a.hasOwnProperty('sentiment')) {
        const needle = document.getElementById('sentiment-needle');
        let pct = 50 + (a.sentiment * 50);
        pct = Math.max(0, Math.min(100, pct));
        if (needle) needle.style.left = pct + '%';
    }

    // --- PREMIUM USERS: REGIME BADGE ---
    const elRegime = document.getElementById('ai-regime');
    if (elRegime && a.regime) {
        elRegime.innerText = a.regime;
        elRegime.style.color = (a.regime === "TRENDING") ? "#0ecb81" : ((a.regime === "VOLATILE") ? "#f6465d" : "#eebb00");
        elRegime.style.border = `1px solid ${elRegime.style.color}`;
    }

    // --- 2. PROFESSIONAL TRADE PLAN Box (Dynamic) ---
    const activeBtn = document.querySelector('.ai-tab.active');
    const mode = activeBtn ? activeBtn.getAttribute('onclick').match(/'([^']+)'/)[1] : 'intraday';
    renderTradePlan(mode);

    // --- PRO UPGRADE: TRIGGER AUDIO ---
    if (a.intraday && (a.intraday.signal === 'BUY' || a.intraday.signal === 'SELL')) {
        AudioAlerts.play(a.intraday.signal);
    }
}

// Function to render the bottom Professional Plan Box
// Function to render the bottom Professional Plan Box
function renderTradePlan(mode) {
    if (!CURRENT_ANALYSIS || !CURRENT_ANALYSIS[mode]) return;

    const sig = CURRENT_ANALYSIS[mode];
    const d = sig.details;
    const breakdownBox = document.getElementById('ai-breakdown-box');

    if (!breakdownBox) return;

    // Handle WAIT or Insufficient Data
    if (!d || sig.signal === "WAIT") {
        breakdownBox.innerHTML = `
            <div style="height:100%; display:flex; flex-direction:column; justify-content:center; align-items:center; color:#555;">
                <div style="font-size:0.9rem; margin-bottom:5px;">Waiting for high-confidence setup...</div>
                <div style="font-size:0.75rem;">Market is ${CURRENT_ANALYSIS.regime || '...'}</div>
            </div>
        `;
        return;
    }

    // Calculations: Ensure numbers
    // Parsing Fix for Indian Market Commas
    const entryVal = parseFloat(String(d.entry).replace(/,/g, ''));
    const slVal = parseFloat(String(d.support).replace(/,/g, ''));
    const tgVal = parseFloat(String(d.resistance).replace(/,/g, ''));

    const slDist = (Math.abs(entryVal - slVal)).toFixed(2);
    const tgDist = (Math.abs(tgVal - entryVal)).toFixed(2);
    const rr = (parseFloat(slDist) > 0) ? (parseFloat(tgDist) / parseFloat(slDist)).toFixed(1) : 0;

    // --- AI POSITION SIZING CALCULATION ---
    const userCap = parseFloat(localStorage.getItem('userCapital')) || 0;
    const userRisk = parseFloat(localStorage.getItem('userRisk')) || 0.01; // Default 1%

    let sizingHtml = '';
    if (userCap > 0 && parseFloat(slDist) > 0) {
        const riskAmt = userCap * userRisk;
        // --- LEVEL 6: POSITION SIZING AI (Confidence Modifier) ---
        // Final Size = Base Size * Confidence
        const conf = sig.conf || 50;
        const confModifier = Math.max(0.1, conf / 100); // Min 10% size
        let qty = Math.floor((riskAmt / parseFloat(slDist)) * confModifier);

        // PERSISTENCE LOGIC: Check if user is currently editing
        const activeInput = document.getElementById('userQtyInput');
        // Correct check:
        if (activeInput && document.activeElement === activeInput) {
            const userVal = parseFloat(activeInput.value);
            if (!isNaN(userVal)) {
                qty = userVal; // Override calculated AI suggested Qty with User's Manual Qty
            }
        }

        let profitAmt = qty * parseFloat(tgDist);

        // UX: Interactive Input
        window.recalcRisk = function (el) {
            const newQty = parseFloat(el.value) || 0;
            const newRisk = newQty * parseFloat(slDist);
            const newProfit = newQty * parseFloat(tgDist);

            document.getElementById('dynRisk').innerText = `-${getCurrency()}${newRisk.toFixed(0)}`;
            document.getElementById('dynProfit').innerText = `+${getCurrency()}${newProfit.toFixed(0)}`;
        };

        if (qty <= 0) {
            // Only show warning if NOT editing manually (if editing, user might type '0' or '1')
            // Actually if user types 0, show 0.
            sizingHtml = `
            <div style="margin-top:10px; padding-top:8px; border-top:1px dashed #2b313f;">
                 <div style="font-size:0.9rem; font-weight:bold; color:#eebb00; text-align:center; margin-bottom:5px;">LOW SIGNAL / AVOID</div>
                 <div style="font-size:0.75rem; color:#aaa; text-align:center;">Risk/Reward not favorable</div>
            </div>`;
        } else {
            sizingHtml = `
            <div style="margin-top:10px; padding-top:8px; border-top:1px dashed #2b313f;">
                <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:5px;">
                   <div style="text-align:left; width:60%;">
                        <div style="font-size:0.7rem; color:#aaa;">QTY (EDITABLE)</div>
                        <input type="number" id="userQtyInput" value="${qty}" oninput="recalcRisk(this)" 
                            style="width:100px; background:rgba(255,255,255,0.1); border:1px solid #555; color:#fff; font-weight:bold; padding:2px 5px; border-radius:3px;">
                        <span style="font-size:0.7rem; font-weight:normal; color:#777;">shares</span>
                   </div>
                </div>
                <div style="display:grid; grid-template-columns: 1fr 1fr; gap:10px;">
                   <div style="background:rgba(246, 70, 93, 0.1); padding:4px 8px; border-radius:3px;">
                        <div style="font-size:0.65rem; color:#ccc;">RISK</div>
                        <div id="dynRisk" style="font-size:0.9rem; font-weight:bold; color:#f6465d;">-${getCurrency()}${((qty * parseFloat(slDist)) || 0).toFixed(0)}</div>
                   </div>
                   <div style="background:rgba(14, 203, 129, 0.1); padding:4px 8px; border-radius:3px;">
                        <div id="dynProfit" style="font-size:0.9rem; font-weight:bold; color:#0ecb81;">+${getCurrency()}${profitAmt.toFixed(0)}</div>
                   </div>
                </div>
            </div>
        `;
        }
    } else if (userCap === 0) {
        sizingHtml = `
            <div style="margin-top:10px; padding-top:5px; border-top:1px dashed #2b313f; text-align:center;">
                <span style="font-size:0.7rem; color:#555; cursor:pointer;" onclick="openSettingsModal()">⚠️ Set Capital in Settings for Sizing</span>
            </div>
        `;
    }


    breakdownBox.innerHTML = `
        <div style="margin-bottom:8px; display:flex; justify-content:space-between; align-items:center;">
            <span style="color:#fff; font-weight:700; font-size:0.9rem;">${sig.trade_type?.toUpperCase()} PLAN</span>
            <span style="font-size:0.75rem; color:#7e879a;">Hold: ${sig.hold_time}</span>
        </div>

        <div style="display:grid; grid-template-columns: 1fr 1fr; gap:8px; margin-bottom:10px;">
            <div style="background:rgba(255,255,255,0.05); padding:8px; border-radius:4px;">
                <div style="font-size:0.7rem; color:#7e879a;">ENTRY</div>
                <div style="font-size:1rem; font-weight:600; color:#fff;">${getCurrency()}${d.entry}</div>
            </div>
            <div style="background:rgba(255,255,255,0.05); padding:8px; border-radius:4px;">
                <div style="font-size:0.7rem; color:#7e879a;">${mode === 'longterm' ? 'SUPPORT' : 'ATR RISK'}</div>
                <div style="font-size:1rem; font-weight:600; color:#eebb00;">${d.atr === 'N/A' ? d.support : d.atr}</div>
            </div>
        </div>

        <div style="display:grid; grid-template-columns: 1fr 1fr; gap:8px;">
            <div style="background:rgba(246, 70, 93, 0.1); padding:8px; border-radius:4px; border:1px solid rgba(246, 70, 93, 0.3);">
                <div style="font-size:0.7rem; color:#f6465d;">STOPLOSS</div>
                <div style="font-size:1rem; font-weight:600; color:#f6465d;">${getCurrency()}${d.support}</div>
                <div style="font-size:0.65rem; color:#f6465d;">-${slDist} pts</div>
            </div>
            <div style="background:rgba(14, 203, 129, 0.1); padding:8px; border-radius:4px; border:1px solid rgba(14, 203, 129, 0.3);">
                <div style="font-size:0.7rem; color:#0ecb81;">TARGET</div>
                <div style="font-size:1rem; font-weight:600; color:#0ecb81;">${getCurrency()}${d.resistance}</div>
                <div style="font-size:0.65rem; color:#0ecb81;">+${tgDist} pts</div>
            </div>
        </div>
        
        <div style="margin-top:8px; font-size:0.65rem; color:#7e879a; text-align:center;">
             Risk Reward: 1:${rr} | ${sig.invalidation ? sig.invalidation[0] : 'Structure Fail'}
        </div>

        ${sizingHtml}
    `;
}

// Global Switch Function triggers re-render
window.switchLanguage = function () {
    const activeTab = document.querySelector('.ai-tab.active');
    const mode = activeTab ? activeTab.innerText.toLowerCase().replace('-', '') : 'intraday';
    if (state.data && state.data.length > 0) fetchMarket(); // Easiest way to refresh UI
    fetchNews(); // Refresh News immediately with new language
}

// Save Position Settings (Global)
window.savePositionSettings = function () {
    const cap = document.getElementById('userCapital').value;
    const risk = document.getElementById('userRisk').value;

    if (cap) {
        localStorage.setItem('userCapital', cap);
        localStorage.setItem('userRisk', risk);
        alert('Trading Preferences Saved!');

        // Refresh UI immediately if data exists
        if (state.data.length > 0) {
            const activeTab = document.querySelector('.ai-tab.active');
            const mode = activeTab ? activeTab.innerText.toLowerCase().replace('-', '') : 'intraday';
            renderTradePlan(mode);
        }
        document.getElementById('settingsModal').style.display = 'none';
    } else {
        alert('Please enter a valid capital amount.');
    }
}

// Settings Modal Logic
function openSettingsModal() {
    const modal = document.getElementById('settingsModal');
    if (!modal) return;

    // Load Saved Values
    const cap = localStorage.getItem('userCapital');
    const risk = localStorage.getItem('userRisk');
    if (cap) document.getElementById('userCapital').value = cap;
    if (risk) document.getElementById('userRisk').value = risk;

    const elPlan = document.getElementById('userPlanDisplay');
    if (elPlan) {
        elPlan.innerText = USER_INFO.plan.toUpperCase();
        elPlan.style.color = USER_INFO.plan === 'premium' ? '#0ecb81' : '#3b82f6';
    }
    const btnUpgrade = document.getElementById('btnUpgrade');
    if (btnUpgrade) {
        if (USER_INFO.plan === 'premium') {
            btnUpgrade.innerText = "Premium Active (Lifetime)";
            btnUpgrade.disabled = true;
            btnUpgrade.style.background = '#2b313f';
        } else {
            btnUpgrade.innerText = "Upgrade to Premium";
        }
    }
    modal.style.display = 'flex';
}

// Wire Settings Form
const settingsForm = document.getElementById('changePassForm');
if (settingsForm) {
    settingsForm.addEventListener('submit', async function (e) {
        e.preventDefault();
        const newPass = document.getElementById('newPass').value;
        try {
            const res = await fetch('api.php?action=change_own_password', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'X-API-TOKEN': API_TOKEN
                },
                body: `password=${newPass}`
            });
            const data = await res.json();
            if (data.success) {
                alert('Password Updated Successfully!');
                document.getElementById('settingsModal').style.display = 'none';
            } else {
                alert('Error: ' + data.error);
            }
        } catch (e) { console.error(e); }
    });
}
function onMouseDown(e) { state.drag.active = true; state.drag.x = e.clientX; state.drag.y = e.clientY; canvas.style.cursor = 'grabbing'; }
function onMouseUp() { state.drag.active = false; canvas.style.cursor = 'crosshair'; }
function onMouseMove(e) {
    const rect = canvas.getBoundingClientRect();
    state.mouse.x = e.clientX - rect.left;
    state.mouse.y = e.clientY - rect.top;

    if (state.drag.active) {
        const dx = e.clientX - state.drag.x;
        const width = canvas.width - 50; // Use chart width approx
        const candleW = width / state.zoom;
        state.offset += dx / candleW;
        if (state.offset < 0) state.offset = 0;
        state.drag.x = e.clientX;
        state.drag.y = e.clientY;
    }
    draw();
}
function onWheel(e) {
    e.preventDefault();
    const dir = Math.sign(e.deltaY);
    const speed = Math.max(1, Math.floor(state.zoom * 0.1));
    state.zoom += dir * speed;
    if (state.zoom < 10) state.zoom = 10;
    if (state.zoom > 1000) state.zoom = 1000;
    draw();
}
function resize() {
    const parent = canvas.parentElement;
    if (parent && parent.clientWidth > 0) {
        const dpr = window.devicePixelRatio || 1;
        canvas.width = parent.clientWidth * dpr;
        canvas.height = parent.clientHeight * dpr;
        canvas.style.width = parent.clientWidth + 'px';
        canvas.style.height = parent.clientHeight + 'px';
        ctx.scale(dpr, dpr);
        draw();
    }
}
function draw() {
    const w = parseInt(canvas.style.width) || canvas.width;
    const h = parseInt(canvas.style.height) || canvas.height;
    ctx.clearRect(0, 0, w, h);

    if (!state.data || state.data.length === 0) {
        ctx.fillStyle = THEME.text; ctx.textAlign = 'center';
        ctx.fillText("Loading Data...", w / 2, h / 2);
        return;
    }

    const padB = 30; const padR = 60; const chartW = w - padR; const chartH = h - padB;
    const gapRatio = 0.9;
    const total = state.data.length; const end = total - Math.floor(state.offset);
    const start = Math.max(0, end - state.zoom); const viewData = state.data.slice(start, end);
    if (viewData.length < 2) return;

    let minP = Infinity, maxP = -Infinity, maxV = 0;
    viewData.forEach(c => { if (c.l < minP) minP = c.l; if (c.h > maxP) maxP = c.h; if (c.v > maxV) maxV = c.v; });
    if (maxV === 0) maxV = 1;
    const midP = (minP + maxP) / 2; const halfRange = ((maxP - minP) / 2) / state.vZoom;
    minP = midP - halfRange; maxP = midP + halfRange;

    const candleW = (chartW * gapRatio) / viewData.length;
    const getX = (i) => i * candleW; const getY = (p) => chartH - ((p - minP) / (maxP - minP)) * chartH;
    const getVolY = (v) => chartH - ((v / maxV) * (chartH * 0.2));

    ctx.strokeStyle = THEME.grid; ctx.lineWidth = 1; ctx.setLineDash([2, 5]); ctx.beginPath();
    ctx.textAlign = 'left'; ctx.fillStyle = THEME.text; ctx.font = '11px sans-serif';
    for (let i = 0; i < 6; i++) {
        const p = minP + (i * ((maxP - minP) / 5)); const y = getY(p);
        if (y < 0 || y > chartH) continue;
        ctx.moveTo(0, y); ctx.lineTo(chartW, y); ctx.fillText(p.toFixed(2), chartW + 5, y + 4);
    }
    const timeStep = Math.max(5, Math.floor(viewData.length / 5)); ctx.textAlign = 'center';
    viewData.forEach((c, i) => {
        if (i % timeStep === 0) {
            const x = getX(i) + candleW / 2; ctx.moveTo(x, 0); ctx.lineTo(x, chartH);
            const d = new Date(c.t * 1000);
            let lbl = (state.interval.includes('d') || state.zoom > 300) ? `${d.getDate()}/${d.getMonth() + 1}` : `${d.getHours()}:${String(d.getMinutes()).padStart(2, '0')}`;
            if (state.interval === '1y') lbl = d.getFullYear();
            ctx.fillText(lbl, x, chartH + 15);
        }
    });
    ctx.stroke(); ctx.setLineDash([]);
    viewData.forEach((c, i) => {
        const x = getX(i); const vH = (chartH - getVolY(c.v)); const y = chartH - vH;
        const isUp = c.c >= c.o;
        ctx.fillStyle = isUp ? 'rgba(14, 203, 129, 0.15)' : 'rgba(246, 70, 93, 0.15)';
        ctx.fillRect(x + 1, y, candleW - 2, vH);
    });
    viewData.forEach((c, i) => {
        const x = getX(i); const cx = x + candleW / 2;
        const yO = getY(c.o); const yC = getY(c.c); const yH = getY(c.h); const yL = getY(c.l);
        const isUp = c.c >= c.o;
        ctx.fillStyle = isUp ? THEME.up : THEME.down; ctx.strokeStyle = ctx.fillStyle;
        if (yH > chartH && yL > chartH) return; if (yH < 0 && yL < 0) return;
        ctx.beginPath(); ctx.moveTo(cx, yH); ctx.lineTo(cx, yL); ctx.stroke();
        const bodyH = Math.max(Math.abs(yC - yO), 1);
        ctx.fillRect(x + (candleW * 0.15), Math.min(yO, yC), candleW * 0.7, bodyH);
    });
    if (state.offset === 0 && viewData.length > 0) {
        const last = viewData[viewData.length - 1]; const y = getY(last.c);
        const color = last.c >= last.o ? THEME.up : THEME.down;
        ctx.strokeStyle = color; ctx.setLineDash([4, 4]); ctx.lineWidth = 1; ctx.beginPath();
        const lastX = getX(viewData.length - 1) + candleW / 2;
        if (lastX < chartW) { ctx.moveTo(lastX, y); ctx.lineTo(w, y); ctx.stroke(); }
        ctx.setLineDash([]);
        ctx.fillStyle = '#1b2029'; ctx.fillRect(chartW, y - 10, padR, 20);
        ctx.lineWidth = 1; ctx.strokeStyle = color; ctx.strokeRect(chartW, y - 10, padR - 1, 20);
        ctx.fillStyle = color; ctx.textAlign = 'left'; ctx.font = 'bold 11px sans-serif';
        ctx.fillText(last.c.toFixed(2), chartW + 5, y + 4);
    }
    const mouseIdx = Math.floor(state.mouse.x / candleW);
    if (mouseIdx >= 0 && mouseIdx < viewData.length && state.mouse.x < chartW * gapRatio && state.mouse.y < chartH) {
        const c = viewData[mouseIdx]; const x = getX(mouseIdx) + candleW / 2;
        ctx.strokeStyle = '#ffffff'; ctx.setLineDash([5, 5]); ctx.beginPath();
        ctx.moveTo(x, 0); ctx.lineTo(x, chartH); ctx.moveTo(0, state.mouse.y); ctx.lineTo(chartW, state.mouse.y);
        ctx.stroke(); ctx.setLineDash([]);
        ctx.fillStyle = 'rgba(0,0,0,0.8)'; ctx.fillRect(5, 5, 230, 20);
        ctx.fillStyle = '#fff'; ctx.textAlign = 'left';
        const d = new Date(c.t * 1000);
        ctx.fillText(`${d.toLocaleString()}  C: ${c.c}`, 10, 18);
    }

    // --- PRO UPGRADE: VISUAL RISK MANAGEMENT LINES ---
    if (CURRENT_ANALYSIS && CURRENT_ANALYSIS.intraday) {
        const d = CURRENT_ANALYSIS.intraday.details;
        const sig = CURRENT_ANALYSIS.intraday.signal;

        if ((sig === 'BUY' || sig === 'SELL') && d && d.support && d.resistance) {
            const sl = parseFloat(d.support);
            const tp = parseFloat(d.resistance);
            const entry = parseFloat(d.entry);

            // Helper to get Y coordinate (Using existing getY in scope)

            // 1. Draw Stoploss (Red)
            const ySL = getY(sl);
            ctx.beginPath();
            ctx.strokeStyle = '#f6465d';
            ctx.lineWidth = 1.5;
            ctx.setLineDash([5, 3]);
            ctx.moveTo(0, ySL);
            ctx.lineTo(chartW, ySL);
            ctx.stroke();
            ctx.setLineDash([]);

            // Label SL
            ctx.fillStyle = '#f6465d';
            ctx.font = '11px sans-serif';
            ctx.textAlign = 'right';
            ctx.fillText(`SL: ${sl}`, chartW - 5, ySL - 5);

            // 2. Draw Target (Green)
            const yTP = getY(tp);
            ctx.beginPath();
            ctx.strokeStyle = '#0ecb81'; // Green
            ctx.lineWidth = 1.5;
            ctx.setLineDash([5, 3]);
            ctx.moveTo(0, yTP);
            ctx.lineTo(chartW, yTP);
            ctx.stroke();
            ctx.setLineDash([]);

            // Label TP
            ctx.fillStyle = '#0ecb81';
            ctx.textAlign = 'right';
            ctx.fillText(`TP: ${tp}`, chartW - 5, yTP - 5);

            // 3. Draw Entry (Blue/White)
            const yEntry = getY(entry);
            ctx.beginPath();
            ctx.strokeStyle = '#3b82f6'; // Blue
            ctx.lineWidth = 1;
            ctx.setLineDash([2, 5]);
            ctx.moveTo(0, yEntry);
            ctx.lineTo(chartW, yEntry);
            ctx.stroke();
            ctx.setLineDash([]);
            ctx.fillStyle = '#3b82f6';
            ctx.fillText(`ENTRY: ${entry}`, chartW - 5, yEntry - 5);

            // 4. Fill Risk Zone (Red shade between Entry and SL)
            ctx.fillStyle = 'rgba(246, 70, 93, 0.05)';
            ctx.fillRect(0, Math.min(yEntry, ySL), chartW, Math.abs(yEntry - ySL));

            // 5. Fill Reward Zone (Green shade between Entry and TP)
            ctx.fillStyle = 'rgba(14, 203, 129, 0.05)';
            ctx.fillRect(0, Math.min(yEntry, yTP), chartW, Math.abs(yEntry - yTP));
        }
    }
}
init();
// app.js V48 continued
function initTicker() {
    console.log("Starting Live Scanner (Immediate)...");
    runScanner();
    setInterval(runScanner, 60000);
}
function runScanner() {
    // Pass current market state to API
    fetch(`api.php?action=scan_market&limit=5&market=${state.market || 'IN'}&interval=${state.interval}`, { headers: API_HEADERS })
        .then(response => response.json())
        .then(data => {
            if (data.intraday) {
                renderTickerRow('ticker-intraday', data.intraday);
                checkSureShot(data.intraday);
            }
            if (data.swing) {
                renderTickerRow('ticker-swing', data.swing);
                checkSureShot(data.swing);
            }
            if (data.longterm) {
                renderTickerRow('ticker-longterm', data.longterm);
                checkSureShot(data.longterm);
            }
        })
        .catch(err => console.error("Scanner Error:", err));
    // visual indicator for debug
    const indicator = document.getElementById('ai-regime');
    if (indicator) indicator.innerText = 'Offline (Retrying...)';
}
function renderTickerRow(id, data) {
    const wrap = document.querySelector(`#${id} .ticker-move`);
    if (!wrap) return;
    const priority = { 'STRONG BUY': 1, 'STRONG SELL': 2, 'BUY': 3, 'ACCUMULATE': 3, 'INVEST': 1, 'SIP MODE': 3, 'SELL': 4, 'EXIT': 2, 'WAIT': 5, 'HOLD': 5 };
    const sortedData = [...data].sort((a, b) => (priority[a.sig] || 5) - (priority[b.sig] || 5));
    let html = ""; let loopData = [...sortedData];
    while (loopData.length < 15) { loopData = [...loopData, ...sortedData]; }
    // Level 10: Store for Brain View
    if (!window.tickerData) window.tickerData = {};

    loopData.forEach(item => {
        // Store item data by symbol for lookup
        window.tickerData[item.s] = item;

        let colorClass = 'rec-' + item.type;
        // Modified OnClick to trigger Brain View
        html += `<div class="ticker-item" onclick="showBrain('${item.s}')" style="cursor:pointer;"><div class="ticker-row-top"><span class="ticker-label">${item.n}</span><span class="ticker-rec ${colorClass}">${item.sig}</span></div><div class="ticker-reason">${item.reason}</div></div>`;
    });
    wrap.innerHTML = html;
}

// --- LEVEL 10: BRAIN VIEW MODAL ---
// Inject Modal HTML once
if (!document.getElementById('brainModal')) {
    const modalHTML = `
    <div id="brainModal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.8); z-index:9999; align-items:center; justify-content:center; backdrop-filter:blur(5px);">
        <div style="background:#1b2029; width:90%; max-width:400px; border-radius:12px; border:1px solid #333; overflow:hidden; box-shadow:0 10px 40px rgba(0,0,0,0.5);">
            <div style="padding:15px; border-bottom:1px solid #333; display:flex; justify-content:space-between; align-items:center; background:#161b22;">
                <h3 id="brainTitle" style="margin:0; font-size:16px; color:#fff;">AI Analysis</h3>
                <button onclick="document.getElementById('brainModal').style.display='none'" style="background:none; border:none; color:#888; cursor:pointer; font-size:20px;">&times;</button>
            </div>
            <div id="brainBody" style="padding:20px; color:#ccc; font-size:13px; line-height:1.6;">
                <!-- Content -->
            </div>
            <div style="padding:15px; background:#161b22; text-align:center;">
                <button onclick="loadChartFromBrain()" style="background:#2962ff; color:#fff; border:none; padding:8px 20px; border-radius:4px; cursor:pointer;">Open Chart</button>
            </div>
        </div>
    </div>`;
    document.body.insertAdjacentHTML('beforeend', modalHTML);
}

let currentBrainSymbol = '';

function showBrain(symbol) {
    const item = window.tickerData[symbol];
    if (!item) return;

    currentBrainSymbol = symbol;

    const brain = item.brain || {};
    const conf = item.conf || 0;

    // Progress Bar Logic
    let barColor = conf > 60 ? '#00e676' : (conf < 40 ? '#ff1744' : '#ffea00');

    let detailsHTML = `
        <div style="display:flex; justify-content:space-between; margin-bottom:15px;">
            <span style="color:#fff; font-size:18px; font-weight:bold;">${item.sig}</span>
            <span style="color:${barColor}; font-weight:bold;">${conf}% Confidence</span>
        </div>
        <div style="background:#333; height:4px; width:100%; margin-bottom:20px; border-radius:2px;">
            <div style="width:${conf}%; background:${barColor}; height:100%; border-radius:2px;"></div>
        </div>
        <div style="display:flex; flex-direction:column; gap:10px;">
    `;

    // Strategy Breakdown
    for (const [key, val] of Object.entries(brain)) {
        if (!val.desc) continue;
        let icon = val.weight > 0 ? '✅' : (val.weight < 0 ? '⚠️' : 'ℹ️');
        let color = val.weight > 0 ? '#00e676' : (val.weight < 0 ? '#ff1744' : '#888');

        detailsHTML += `
            <div style="display:flex; align-items:flex-start; gap:10px; border-bottom:1px solid #2a2a2a; padding-bottom:8px;">
                <span style="font-size:14px;">${icon}</span>
                <div>
                    <div style="color:#fff; font-weight:bold; font-size:13px;">${val.value}</div>
                    <div style="color:#888; font-size:11px;">${val.desc}</div>
                </div>
            </div>
        `;
    }
    detailsHTML += `</div>`;

    document.getElementById('brainTitle').innerText = `${item.n} (${symbol})`;
    document.getElementById('brainBody').innerHTML = detailsHTML;
    document.getElementById('brainModal').style.display = 'flex';
}

function loadChartFromBrain() {
    document.getElementById('brainModal').style.display = 'none';
    document.getElementById('headerSearch').value = currentBrainSymbol;
    document.getElementById('btnSearch').click();
}
// --- GENERAL NEWS ROTATION LOGIC ---
let newsList = [];
let newsIdx = 0;

function initGeneralNews() {
    // ONE TIME Fetch or Periodic Fetch
    fetchNews();
    // Refresh News List every 10 mins
    setInterval(fetchNews, 600000);
    // Rotate every 1 min (60000ms)
    setInterval(rotateNews, 30000);
}

function fetchNews() {
    const langSelect = document.getElementById('langSelect');
    const lang = langSelect ? langSelect.value : 'en';
    fetch(`api.php?action=get_market_news&lang=${lang}`, { headers: API_HEADERS })
        .then(res => res.json())
        .then(data => {
            if (data.news && data.news.length > 0) {
                newsList = data.news;
                rotateNews(); // Show first immediately
            }
        })
        .catch(e => console.error("News Fetch Error", e));
}

function rotateNews() {
    if (newsList.length === 0) return;

    const footerReason = document.getElementById('aiFooterReason');
    const footerTitle = document.getElementById('aiFooterTitle');

    if (!footerReason) return;

    // Fade Out
    footerReason.style.transition = 'opacity 0.5s';
    footerReason.style.opacity = '0';

    setTimeout(() => {
        // Change Text
        const item = newsList[newsIdx];
        footerReason.innerHTML = `<span style="color:#0ecb81; font-weight:800; font-size:0.9rem;">[NEWS]</span> <span style="color:#ffffff; font-size:0.9rem; font-weight:500;">${item.title}</span>`;
        if (footerTitle) footerTitle.innerText = "MARKET UPDATES";
        if (footerTitle) footerTitle.style.color = "#3b82f6"; // Blue for News

        // Next Index
        newsIdx = (newsIdx + 1) % newsList.length;

        // Fade In
        footerReason.style.opacity = '1';
    }, 500);
}

// Inject Sure Shot Modal
if (!document.getElementById('sureShotModal')) {
    const modal = `
        <div id="sureShotModal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.9); z-index:10000; align-items:center; justify-content:center; backdrop-filter:blur(10px);">
            <div style="background:linear-gradient(135deg, #1e222d 0%, #000 100%); width:90%; max-width:450px; border-radius:20px; border:2px solid #0ecb81; box-shadow:0 0 50px rgba(14,203,129,0.3); text-align:center; padding:30px; position:relative; overflow:hidden;">
                <div style="position:absolute; top:-20px; left:-20px; width:100px; height:100px; background:#0ecb81; filter:blur(60px); opacity:0.3;"></div>
                
                <h1 style="color:#0ecb81; font-size:3rem; margin:0; text-shadow:0 0 20px rgba(14,203,129,0.5);">100%</h1>
                <h2 style="color:#fff; font-size:1.5rem; margin:10px 0;">SURE SHOT ENTRY</h2>
                
                <div id="sureShotSymbol" style="font-size:2rem; font-weight:900; color:#fff; margin:20px 0; padding:10px; background:rgba(255,255,255,0.05); border-radius:10px;">
                    SYMBOL
                </div>
                
                <div id="sureShotReason" style="color:#aaa; font-size:1rem; margin-bottom:30px;">
                    Reasoning...
                </div>
                
                <button onclick="document.getElementById('sureShotModal').style.display='none'" 
                   style="background:#0ecb81; color:#000; border:none; padding:15px 40px; font-size:1.2rem; font-weight:bold; border-radius:50px; cursor:pointer; box-shadow:0 5px 20px rgba(14,203,129,0.4); transition:transform 0.2s;">
                    ACKNOWLEDGE 🚀
                </button>
            </div>
        </div>`;
    document.body.insertAdjacentHTML('beforeend', modal);
}

let sureShotSeen = new Set();
let sureShotQueue = [];
let lastSureShotTime = 0;
let globalMaxConf = 0;

// PERIODIC QUEUE PROCESSOR
setInterval(() => {
    const now = Date.now();
    // 60 Seconds Cooldown (First one can be faster, e.g. 5s after load)
    const cooldown = (lastSureShotTime === 0) ? 5000 : 60000;

    if (now - lastSureShotTime > cooldown && sureShotQueue.length > 0) {
        sureShotQueue.sort((a, b) => b.conf - a.conf);
        const topItem = sureShotQueue.shift();
        showSureShot(topItem);
        lastSureShotTime = now;
        if (window.AudioAlerts && topItem.sig) window.AudioAlerts.play(topItem.sig);
    }
}, 1000);

// Universal Sure Shot Checker
function checkSureShot(data) {
    if (!data) return;
    const list = Array.isArray(data) ? data : [data];

    // Fix: Only process signals for the ACTIVE market
    // If state.market is set, ensure symbol matches the market type
    const activeMarket = state.market || 'IN';

    list.forEach(item => {
        const signal = item.sig || item.signal;
        const conf = item.conf; // Should be number
        const symbol = item.s || state.symbol;
        const name = item.n || symbol;
        const reason = item.reason || item.text_en || "High Confidence Signal";

        // Market Filter
        const isUS = !symbol.includes('.NS') && !symbol.includes('.BO');
        const isIN = symbol.includes('.NS') || symbol.includes('.BO');

        if (activeMarket === 'IN' && isUS) return; // Skip US stocks if IN market active
        if (activeMarket === 'US' && isIN) return; // Skip IN stocks if US market active

        // TRACK MAX CONFIDENCE FOR USER
        if (conf > globalMaxConf) {
            globalMaxConf = conf;
            const statusEl = document.getElementById('ai-regime');
            if (statusEl) {
                statusEl.innerHTML = `${state.market || 'MARKET'} <span style="font-size:0.8rem; color:#aaa;">(Max: ${globalMaxConf}%)</span>`;
            }
        }

        // 90% Threshold
        if (conf >= 90 && signal !== 'WAIT' && signal !== 'HOLD' && !sureShotSeen.has(symbol)) {
            const exists = sureShotQueue.find(q => q.s === symbol);
            if (!exists) {
                sureShotQueue.push({ s: symbol, n: name, sig: signal, reason: reason, conf: conf });
                sureShotSeen.add(symbol);
                console.log(`Queued Sure Shot: ${symbol} (${conf}%)`);
            }
        }
    });
}

// TEST TRIGGER: Click "AI SAYS" title to force a Test Popup
document.addEventListener('DOMContentLoaded', () => {
    setTimeout(() => {
        const header = document.querySelector('.section-header');
        if (header) {
            header.style.cursor = 'pointer';
            header.title = "Click to Test Popup";
            header.onclick = () => {
                // alert("Testing Popup Mechanism...");
                console.log("Triggering Test Popup");
                showSureShot({ s: 'TEST.NS', n: 'TEST SIGNAL', sig: 'STRONG BUY', reason: 'User Requested Test', conf: 100 });
            };
        }
    }, 2000);
});

function showSureShot(item) {
    const popup = document.getElementById('sureShotPopup');
    if (!popup) return;

    // Color Logic
    const isBuy = item.sig.includes('BUY') || item.sig.includes('ACCUM');
    const color = isBuy ? '#0ecb81' : '#f6465d';

    document.getElementById('ssTitle').style.color = color;
    document.getElementById('ssSymbol').innerText = item.n;
    document.getElementById('ssSymbol').style.color = "#fff"; // Keep white for readability

    const confEl = document.getElementById('ssConf');
    confEl.innerText = item.conf + "%";
    confEl.style.color = color;
    confEl.style.background = isBuy ? 'rgba(14,203,129,0.1)' : 'rgba(246,70,93,0.1)';

    const sigEl = document.getElementById('ssSignal');
    sigEl.innerText = item.sig;
    sigEl.style.color = color;

    document.getElementById('ssReason').innerText = item.reason || "AI detected high probability setup.";

    popup.style.display = 'flex';

    // Auto-hide after 15 seconds
    setTimeout(() => {
        popup.style.display = 'none';
    }, 15000);
}

document.addEventListener('DOMContentLoaded', initTicker);
