<?php
// master_debug_tool.php - The "Z to Z" Debugger
// Purpose: Diagnose WHY data is missing, check Server Health, and Validate Integrity.

header('Content-Type: text/html; charset=utf-8');
ini_set('display_errors', 1);
error_reporting(E_ALL);

function pass($msg)
{
    echo "<div style='color:green; margin:5px 0;'>✅ PASS: $msg</div>";
}
function fail($msg)
{
    echo "<div style='color:red; margin:5px 0; font-weight:bold;'>❌ FAIL: $msg</div>";
}
function info($msg)
{
    echo "<div style='color:blue; margin:5px 0;'>ℹ️ INFO: $msg</div>";
}

echo "<h1>🛠️ AutoTrade AI - Master Debug Tool</h1>";
echo "<div style='background:#f5f5f5; padding:20px; border:1px solid #ccc; font-family:monospace;'>";

// 1. SYSTEM BASICS
echo "<h3>1. SERVER ENVIRONMENT</h3>";
echo "PHP Version: " . phpversion() . "<br>";
echo "Server Software: " . $_SERVER['SERVER_SOFTWARE'] . "<br>";

$req_ext = ['curl', 'openssl', 'json', 'mbstring'];
foreach ($req_ext as $ext) {
    if (extension_loaded($ext))
        pass("Extension '$ext' is LOADED.");
    else
        fail("Extension '$ext' is MISSING! Data fetch will fail.");
}

if (ini_get('allow_url_fopen'))
    pass("allow_url_fopen is ENABLED.");
else
    info("allow_url_fopen is DISABLED (cURL is preferred anyway).");

// 2. FILE PERMISSIONS
echo "<h3>2. FILE SYSTEM & PERMISSIONS</h3>";
$dirs = ['data', 'data/charts', 'core', 'ai'];
foreach ($dirs as $d) {
    if (!file_exists($d)) {
        if (mkdir($d, 0777, true))
            pass("Created missing directory: $d");
        else
            fail("Could not create directory: $d");
    }

    if (is_writable($d))
        pass("Directory '$d' is WRITABLE.");
    else
        fail("Directory '$d' is NOT WRITABLE. Cache will fail.");
}

// 3. CORE ENGINE CHECK
echo "<h3>3. CODEBASE INTEGRITY</h3>";
$files = ['index.php', 'api.php', 'app.js', 'style.css', 'core/fetch_engine.php', 'trade_engine.php'];
foreach ($files as $f) {
    if (file_exists($f)) {
        $size = filesize($f);
        if ($size > 100)
            pass("File '$f' exists ($size bytes).");
        else
            fail("File '$f' is suspiciously small ($size bytes). Check content.");
    } else {
        fail("CRITICAL: File '$f' is MISSING.");
    }
}

// 4. LIVE DATA TEST (THE "NO DATA" CHECK)
echo "<h3>4. LIVE DATA CONNECTIVITY (The 'No Data' Test)</h3>";
require_once 'core/fetch_engine.php';

$test_symbols = [
    'Nifty 50' => '^NSEI',
    'Reliance (NSE)' => 'RELIANCE.NS',
    'TCS (NSE)' => 'TCS.NS',
    'US Market (DJI)' => '^DJI' // Test US connectivity
];

foreach ($test_symbols as $name => $sym) {
    echo "<strong>Testing $name ($sym)...</strong><br>";
    $url = "https://query1.finance.yahoo.com/v8/finance/chart/" . urlencode($sym) . "?interval=1d&range=5d";

    $start = microtime(true);
    $res = fetch_url_secure($url);
    $end = microtime(true);
    $dur = round($end - $start, 2);

    if ($res) {
        $json = json_decode($res, true);
        if (isset($json['chart']['result'][0]['meta']['regularMarketPrice'])) {
            $price = $json['chart']['result'][0]['meta']['regularMarketPrice'];
            pass("Got Data for $sym! Price: $price (Time: ${dur}s)");
        } else {
            fail("Connected to Yahoo, but JSON is invalid or empty for $sym. Response size: " . strlen($res));
        }
    } else {
        fail("Could NOT connect to Yahoo Finance for $sym. Network Blocked or Timeout.");
    }
    echo "<hr>";
}

echo "</div>";
echo "<h3>🏁 END DIAGNOSTIC</h3>";
?>