// paper/logic.js - White Theme & Sidebar Update - V36 CURRENCY UPDATE

let currentSymbol = 'RELIANCE.NS';
let currentPrice = 0;
let currentAI = null;
let orderType = 'BUY';
let portfolio = [];
let balanceINR = 0; // Store base balance
let currentCurrency = 'INR'; // INR or USD
const USD_RATE = 84; // Fixed Exchange Rate

// 1. INIT
document.addEventListener('DOMContentLoaded', () => {
    loadPortfolio();
    initSidebar(); // Start Sidebar
    loadSymbol();  // Load default
    initPaperScanner(); // Start Ticker Scanner
    setInterval(updatePrices, 5000); // 5s refresh
    // Ticker Animation Fix
    const style = document.createElement('style');
    style.innerHTML = `
        @keyframes ticker-scroll { 0% { transform: translate3d(0, 0, 0); } 100% { transform: translate3d(-100%, 0, 0); } }
    `;
    document.head.appendChild(style);
});

// 2. PORTFOLIO MANAGER
async function loadPortfolio() {
    try {
        const res = await fetch(`api.php?action=get_portfolio&token=${API_TOKEN}`);
        const data = await res.json();
        balanceINR = parseFloat(data.balance);
        updateWalletUI(); // Update display based on mode
        portfolio = data.positions;
        renderPortfolio();
    } catch (e) { console.error(e); }
}

function updateWalletUI() {
    const el = document.getElementById('walletBalance');
    if (currentCurrency === 'USD') {
        // Convert INR to USD
        const balUSD = balanceINR / USD_RATE;
        el.innerText = '$' + balUSD.toLocaleString(undefined, { maximumFractionDigits: 2 });
        el.style.color = '#0ecb81'; // Green for USD
    } else {
        // INR
        el.innerText = '₹' + balanceINR.toLocaleString();
        el.style.color = '#eebb00'; // Gold for INR
    }
}

function renderPortfolio() {
    const tbody = document.querySelector('#positionsTable tbody');
    tbody.innerHTML = '';
    portfolio.forEach(pos => {
        const isUS = !pos.symbol.includes('.NS');
        const rate = isUS ? USD_RATE : 1;
        const sym = isUS ? '$' : '₹';

        // Prices in DB are stored in NATIVE currency (if we fixed placeOrder) OR INR?
        // Let's assume we store everything in INR in DB to keep math simple?
        // NO, wait. Standard is: DB stores cost in INR.
        // So when rendering, we should convert back to native for display?
        // OR: let's stick to: DB has Entry Price in Native Currency.
        // And Balance is in INR.

        let ltp = (pos.symbol === currentSymbol) ? currentPrice : pos.entry_price;
        // Note: currentPrice is Native.

        const isBuy = pos.side === 'BUY';
        // PnL in Native
        let pnlNative = isBuy ? (ltp - pos.entry_price) * pos.qty : (pos.entry_price - ltp) * pos.qty;

        // PnL in INR (for coloring/total view if needed, but here we show native)
        // const pnlINR = pnlNative * rate;

        const pnlClass = pnlNative >= 0 ? 'pnl-pos' : 'pnl-neg';
        const tr = document.createElement('tr');
        tr.innerHTML = `
            <td>${pos.symbol}</td>
            <td style="color:${isBuy ? '#00c853' : '#ff3d00'}">${pos.side}</td>
            <td>${pos.qty}</td>
            <td>${sym}${pos.entry_price.toFixed(2)}</td>
            <td>${sym}${ltp.toFixed(2)}</td>
            <td class="${pnlClass}">${sym}${pnlNative.toFixed(2)}</td>
            <td><button onclick="closePosition('${pos.id}')" style="background:#ff3d00; color:white; border:none; padding:4px 8px; font-size:0.7rem; border-radius:2px; cursor:pointer;">CLOSE</button></td>
        `;
        tbody.appendChild(tr);
    });
}

// 3. AI & MARKET DATA
async function loadSymbol(manual = false) {
    const sym = document.getElementById('symbolInput') ? document.getElementById('symbolInput').value : currentSymbol;
    if (manual && sym) currentSymbol = sym;

    // Detect Currency Mode based on Symbol
    const isUS = !currentSymbol.includes('.NS');
    // Auto-switch tabs if needed (optional, or just update currency)
    // But user asked "Switching market changes money", so let's respect the Tab

    if (manual || sym !== currentSymbol) {
        document.getElementById('chartSymbolDisplay').innerText = "Loading...";
        document.getElementById('aiFeedList').innerHTML = '<div style="color:#888; font-size:0.8rem; text-align:center; padding:10px;">Analyzing Market Data...</div>';
    }

    try {
        const res = await fetch(`../api.php?action=analyze&symbol=${currentSymbol}&interval=5m&token=${API_TOKEN}`, {
            headers: { 'X-API-TOKEN': API_TOKEN }
        });
        const data = await res.json();
        if (data.error) {
            if (manual) alert(data.error);
            return;
        }
        renderChart(data.candles);

        // Safety Check for Analysis Data (Zomato Fix)
        if (!data.analysis) {
            // Create Dummy Safe Analysis to prevent crash
            data.analysis = {
                intraday: { signal: 'WAIT', conf: 0, text_ml: 'Data insufficient for analysis', text_en: 'Data insufficient' },
                swing: { signal: 'WAIT', conf: 0 },
                longterm: { signal: 'WAIT', conf: 0 }
            };
        }

        // PRICE RECOVERY (If 0, use Last Candle Close)
        let displayPrice = data.price;
        if ((!displayPrice || displayPrice == 0) && data.candles && data.candles.length > 0) {
            displayPrice = data.candles[data.candles.length - 1].c;
            console.log("⚠️ Price Fixed from Candle: " + displayPrice);
        }

        updateAI(data.analysis.intraday, displayPrice, data.analysis);

        // CHECK FOR SURE SHOT (Popup Hook)
        checkSureShot(data.analysis.intraday);

    } catch (e) {
        console.error(e);
        if (manual) alert("Data Error: " + e.message);
    }
}

async function updatePrices() {
    if (currentSymbol) loadSymbol(false);
    loadPortfolio();
}

function updateAI(intraday, price, fullAnalysis) {
    currentPrice = price;
    currentAI = intraday;

    document.getElementById('chartPriceDisplay').innerText = (currentCurrency === 'USD' ? '$' : '₹') + price;
    document.getElementById('chartSymbolDisplay').innerText = currentSymbol;

    if (intraday.signal === 'BUY') setOrderType('BUY');
    if (intraday.signal === 'SELL') setOrderType('SELL');

    const confEl = document.getElementById('aiConfidence');
    if (confEl) {
        confEl.innerText = intraday.conf + '% Confidence';
        confEl.style.color = intraday.conf > 70 ? '#0ecb81' : '#aaa';
    }

    if (intraday.details) {
        const slInput = document.getElementById('orderSL');
        const tgtInput = document.getElementById('orderTgt');

        // Only auto-fill if NOT focused by user (allows manual edit)
        if (slInput && document.activeElement !== slInput && slInput.value == '') slInput.value = intraday.details.support;
        if (tgtInput && document.activeElement !== tgtInput && tgtInput.value == '') tgtInput.value = intraday.details.resistance;

        // Auto Qty Calculation
        // Risk ₹10,000 or $120
        const risk = currentCurrency === 'USD' ? 120 : 10000;
        const dist = Math.abs(price - intraday.details.support);
        if (dist > 0) {
            let qty = Math.floor(risk / dist);
            if (intraday.conf) qty = Math.floor(qty * (intraday.conf / 100));
            if (qty < 1) qty = 1;
            if (document.getElementById('orderQty')) document.getElementById('orderQty').value = qty;
        }
    }

    if (fullAnalysis) {
        updateHorizon('intraday', fullAnalysis.intraday);
        updateHorizon('swing', fullAnalysis.swing);
        updateHorizon('longterm', fullAnalysis.longterm);
        if (fullAnalysis.sentiment) {
            const needle = document.getElementById('sentiment-needle');
            if (needle) {
                let val = 50;
                if (fullAnalysis.sentiment === 'BULLISH') val = 85;
                else if (fullAnalysis.sentiment === 'BEARISH') val = 15;
                needle.style.left = val + '%';
            }
        }
    } else {
        updateHorizon('intraday', intraday);
    }

    const feed = document.getElementById('aiFeedList');
    if (!feed) return;
    const firstMsg = feed.firstElementChild;
    const newMsgContent = `${intraday.signal} ${currentSymbol}`;
    if (firstMsg && firstMsg.innerText.includes(newMsgContent)) return;
    const newCard = `
    <div style="padding:5px; border-bottom:1px solid #333; font-size:10px; color:#888;">
        <span style="color:${intraday.signal === 'BUY' ? '#00c853' : '#ff3d00'}; font-weight:bold;">${intraday.signal}</span> 
        ${currentSymbol} (${intraday.conf}%)
    </div>
    `;
    feed.innerHTML = newCard + feed.innerHTML;
}

function updateHorizon(mode, data) {
    if (!data) return;
    const elSig = document.getElementById('ai-signal-' + mode);
    const elConfBar = document.getElementById('ai-conf-' + mode);
    const elConfVal = document.getElementById('ai-conf-val-' + mode);
    const elDesc = document.getElementById('ai-desc-' + mode);

    if (elSig) {
        elSig.innerText = data.signal;
        elSig.style.color = data.signal === 'BUY' ? '#0ecb81' : (data.signal === 'SELL' ? '#f6465d' : '#eebb00');
    }
    if (elConfBar) elConfBar.style.width = data.conf + '%';
    if (elConfVal) elConfVal.innerText = data.conf + '%';
    if (elDesc) elDesc.innerText = data.text_ml || data.text_en;
}

window.switchAI = function (mode, btn) {
    document.querySelectorAll('.ai-tab').forEach(b => b.classList.remove('active'));
    btn.classList.add('active');
    document.querySelectorAll('.ai-content').forEach(c => c.classList.remove('active'));
    document.getElementById('ai-content-' + mode).classList.add('active');
};

const canvas = document.getElementById('paperChart');
const ctx = canvas.getContext('2d');
const THEME = { bg: '#161b22', grid: '#2a2e39', text: '#cfd3da', up: '#0ecb81', down: '#f6465d' };
const state = { data: [], zoom: 50, offset: 0, vZoom: 1, mouse: { x: 0, y: 0 }, drag: { active: false, x: 0, y: 0 } };

// Stocks Global
let NIFTY_50 = [];
let US_TECH = [];
if (typeof ALL_STOCKS_IN !== 'undefined') NIFTY_50 = ALL_STOCKS_IN;
if (typeof ALL_STOCKS_US !== 'undefined') US_TECH = ALL_STOCKS_US;


let activeList = NIFTY_50;

// Listeners
canvas.addEventListener('mousedown', (e) => { state.drag.active = true; state.drag.x = e.clientX; canvas.style.cursor = 'grabbing'; });
canvas.addEventListener('mouseup', () => { state.drag.active = false; canvas.style.cursor = 'crosshair'; });
canvas.addEventListener('mouseout', () => { state.drag.active = false; });
canvas.addEventListener('mousemove', (e) => {
    const rect = canvas.getBoundingClientRect();
    state.mouse.x = e.clientX - rect.left;
    state.mouse.y = e.clientY - rect.top;
    if (state.drag.active) {
        const dx = e.clientX - state.drag.x;
        state.offset += dx / (canvas.width / state.zoom);
        if (state.offset < 0) state.offset = 0;
        state.drag.x = e.clientX;
        draw();
    }
    draw();
});
canvas.addEventListener('wheel', (e) => {
    e.preventDefault();
    const dir = Math.sign(e.deltaY);
    state.zoom += dir * Math.max(1, Math.floor(state.zoom * 0.1));
    if (state.zoom < 10) state.zoom = 10;
    if (state.zoom > 500) state.zoom = 500;
    draw();
});

// SIDEBAR
function initSidebar() { renderStockList(); }
function switchMarket(market) {
    document.getElementById('btnIN').className = market === 'IN' ? 'active' : '';
    document.getElementById('btnUS').className = market === 'US' ? 'active' : '';

    // UPDATE CURRENCY STATE
    currentCurrency = market === 'US' ? 'USD' : 'INR';
    updateWalletUI();

    activeList = market === 'IN' ? NIFTY_50 : US_TECH;
    document.getElementById('sidebarSearch').value = '';
    renderStockList();

    // SYNC SCANNER INSTANTLY
    // Cancel any existing timeout if we wanted to be fancy, but simple run is enough.
    // The periodic interval will continue, but this gives immediate feedback.
    console.log("⚡ Auto-Switching Scanner to: " + market);
    runPaperScan();
}

function renderStockList() {
    const list = document.getElementById('stockList');
    list.innerHTML = '';
    activeList.forEach(sym => {
        const item = document.createElement('div');
        item.className = 'stock-item';
        let name = sym.replace('.NS', '');
        let desc = sym.includes('.NS') ? 'NSE' : 'NASDAQ';
        item.innerHTML = `<div><div class="sym">${name}</div><div class="desc">${desc}</div></div>`;
        item.onclick = () => loadSymbolWrapper(sym);
        list.appendChild(item);
    });
}
function loadSymbolWrapper(sym) {
    document.getElementById('symbolInput').value = sym;
    loadSymbol(true);
}

// DYNAMIC SEARCH
async function filterStocks() {
    const q = document.getElementById('sidebarSearch').value.toUpperCase();
    const list = document.getElementById('stockList');
    const filtered = activeList.filter(s => s.replace('.NS', '').includes(q));

    list.innerHTML = '';
    filtered.forEach(sym => {
        const item = document.createElement('div');
        item.className = 'stock-item';
        let name = sym.replace('.NS', '');
        let desc = sym.includes('.NS') ? 'NSE' : 'NASDAQ';
        item.innerHTML = `<div><div class="sym">${name}</div><div class="desc">${desc}</div></div>`;
        item.onclick = () => loadSymbolWrapper(sym);
        list.appendChild(item);
    });

    if (filtered.length === 0 && q.length > 2) {
        const btn = document.createElement('div');
        btn.className = 'stock-item';
        btn.style.color = '#eebb00';
        btn.innerHTML = `Check Global Market for '${q}'...`;
        btn.onclick = async () => {
            btn.innerHTML = "Searching...";
            try {
                const res = await fetch(`../api.php?action=search_symbol&query=${q}&token=${API_TOKEN}`);
                const data = await res.json();
                list.innerHTML = '';
                if (data && data.length > 0) {
                    data.forEach(stock => {
                        const item = document.createElement('div');
                        item.className = 'stock-item';
                        item.innerHTML = `<div><div class="sym">${stock.symbol}</div><div class="desc">${stock.name}</div></div>`;
                        item.onclick = () => {
                            if (!activeList.includes(stock.symbol)) activeList.unshift(stock.symbol);
                            loadSymbolWrapper(stock.symbol);
                            renderStockList();
                        };
                        list.appendChild(item);
                    });
                } else {
                    btn.innerHTML = "No results found.";
                }
            } catch (e) {
                btn.innerHTML = "Search Error.";
            }
        };
        list.appendChild(btn);
    }
}

function renderChart(candles) {
    if (!candles || candles.length === 0) return;
    const first = candles[0];
    const kC = first.c ? 'c' : 'Close'; const kH = first.h ? 'h' : 'High'; const kL = first.l ? 'l' : 'Low'; const kO = first.o ? 'o' : 'Open'; const kT = first.t ? 't' : 'Date';
    state.data = candles.map(c => ({ o: parseFloat(c[kO]), h: parseFloat(c[kH]), l: parseFloat(c[kL]), c: parseFloat(c[kC]), t: c[kT] }));
    state.offset = 0;
    resize();
}
function resize() {
    const p = canvas.parentElement;
    if (p) { canvas.width = p.clientWidth; canvas.height = p.clientHeight; draw(); }
}
function draw() {
    const w = canvas.width; const h = canvas.height;
    ctx.clearRect(0, 0, w, h);
    if (state.data.length === 0) { ctx.fillStyle = THEME.text; ctx.textAlign = 'center'; ctx.fillText("Select a stock", w / 2, h / 2); return; }

    const chartW = w - 60;
    const end = state.data.length - Math.floor(state.offset);
    const start = Math.max(0, end - Math.floor(state.zoom));
    const view = state.data.slice(start, end);
    if (view.length < 2) return;

    let min = Infinity, max = -Infinity;
    view.forEach(c => { if (c.l < min) min = c.l; if (c.h > max) max = c.h; });
    const range = max - min; min -= range * 0.1; max += range * 0.1;
    const cW = chartW / view.length;
    const getY = (p) => h - ((p - min) / (max - min)) * h;

    ctx.strokeStyle = THEME.grid; ctx.lineWidth = 1; ctx.beginPath();
    ctx.textAlign = 'left'; ctx.fillStyle = THEME.text; ctx.font = '11px sans-serif';
    for (let i = 0; i < 6; i++) {
        const p = min + (i * ((max - min) / 5)); const y = getY(p);
        ctx.moveTo(0, y); ctx.lineTo(chartW, y); ctx.fillText(p.toFixed(2), chartW + 5, y + 4);
    }
    ctx.stroke();

    view.forEach((c, i) => {
        const x = i * cW; const cx = x + cW / 2;
        const yO = getY(c.o); const yC = getY(c.c); const yH = getY(c.h); const yL = getY(c.l);
        ctx.fillStyle = c.c >= c.o ? THEME.up : THEME.down; ctx.strokeStyle = ctx.fillStyle;
        ctx.beginPath(); ctx.moveTo(cx, yH); ctx.lineTo(cx, yL); ctx.stroke();
        ctx.fillRect(x + 1, Math.min(yO, yC), cW - 2, Math.max(Math.abs(yC - yO), 1));
    });

    if (state.offset === 0) {
        const last = view[view.length - 1]; const y = getY(last.c);
        ctx.strokeStyle = last.c >= last.o ? THEME.up : THEME.down;
        ctx.setLineDash([2, 5]); ctx.beginPath(); ctx.moveTo(0, y); ctx.lineTo(w, y); ctx.stroke(); ctx.setLineDash([]);
        ctx.fillStyle = ctx.strokeStyle; ctx.fillRect(chartW, y - 10, 60, 20);
        ctx.fillStyle = '#fff'; ctx.fillText(last.c.toFixed(2), chartW + 5, y + 4);
    }

    if (state.mouse.x < chartW && state.drag.active === false) {
        const idx = Math.floor(state.mouse.x / cW);
        if (idx >= 0 && idx < view.length) {
            const c = view[idx]; const x = idx * cW + cW / 2;
            ctx.strokeStyle = '#888'; ctx.setLineDash([4, 4]); ctx.beginPath();
            ctx.moveTo(x, 0); ctx.lineTo(x, h); ctx.moveTo(0, state.mouse.y); ctx.lineTo(chartW, state.mouse.y); ctx.stroke(); ctx.setLineDash([]);
            const boxY = h - 60; const boxX = 10;
            ctx.fillStyle = 'rgba(30, 35, 45, 0.95)'; ctx.fillRect(boxX, boxY, 180, 45);
            ctx.strokeStyle = '#555'; ctx.strokeRect(boxX, boxY, 180, 45);
            ctx.fillStyle = '#fff'; ctx.font = '11px monospace';
            ctx.fillText(`${new Date(c.t * 1000).toLocaleDateString()} ${new Date(c.t * 1000).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}`, boxX + 5, boxY + 15);
            ctx.fillText(`O:${c.o} H:${c.h} L:${c.l} C:${c.c}`, boxX + 5, boxY + 30);
        }
    }

    if (currentAI && currentAI.details) {
        const d = currentAI.details;
        if (d.support) {
            const y = getY(parseFloat(d.support));
            ctx.strokeStyle = '#ff3d00'; ctx.lineWidth = 1; ctx.setLineDash([2, 2]);
            ctx.beginPath(); ctx.moveTo(0, y); ctx.lineTo(chartW, y); ctx.stroke();
            ctx.fillStyle = '#ff3d00'; ctx.textAlign = 'right'; ctx.fillText(`SL`, chartW - 5, y - 5);
        }
        if (d.resistance) {
            const y = getY(parseFloat(d.resistance));
            ctx.strokeStyle = '#00c853'; ctx.lineWidth = 1; ctx.setLineDash([2, 2]);
            ctx.beginPath(); ctx.moveTo(0, y); ctx.lineTo(chartW, y); ctx.stroke();
            ctx.fillStyle = '#00c853'; ctx.textAlign = 'right'; ctx.fillText(`TP`, chartW - 5, y - 5);
        }
    }
}

function setOrderType(type) {
    orderType = type;
    document.getElementById('btnBuySide').style.opacity = type === 'BUY' ? 1 : 0.3;
    document.getElementById('btnSellSide').style.opacity = type === 'SELL' ? 1 : 0.3;
    document.getElementById('btnExecute').style.background = type === 'BUY' ? '#0ecb81' : '#f6465d';
}
async function placeOrder() {
    const qty = document.getElementById('orderQty').value;
    const sl = document.getElementById('orderSL').value;
    const tgt = document.getElementById('orderTgt').value;

    // CONVERT PRICE TO INR FOR BACKEND IF USD
    let submitPrice = currentPrice;
    if (currentCurrency === 'USD') {
        submitPrice = currentPrice * USD_RATE;
    }

    const payload = { symbol: currentSymbol, type: orderType, qty: qty, price: submitPrice, sl: sl, tgt: tgt };
    const res = await fetch(`api.php?action=place_order&token=${API_TOKEN}`, { method: 'POST', body: JSON.stringify(payload) });
    const data = await res.json();
    document.getElementById('orderStatus').innerText = data.msg;
    document.getElementById('orderStatus').style.color = data.status === 'success' ? '#0ecb81' : '#f6465d';
    loadPortfolio();
}
async function closePosition(id) {
    if (!confirm("Close this trade?")) return;
    // Note: We send exit_price in INR.
    // So if current is USD, we need to convert.
    let submitPrice = currentPrice;
    if (currentCurrency === 'USD') {
        submitPrice = currentPrice * USD_RATE;
    }

    const payload = { id: id, exit_price: submitPrice };
    const res = await fetch(`api.php?action=close_position&token=${API_TOKEN}`, { method: 'POST', body: JSON.stringify(payload) });
    loadPortfolio();
}
async function resetAccount() {
    if (!confirm("Reset Wallet to ₹100 Crore (Unlimited)?")) return;
    await fetch(`api.php?action=reset_account&token=${API_TOKEN}`);
    loadPortfolio();
}

// --- SURE SHOT POPUP LOGIC (Ported from Main App) ---

// Inject Sure Shot Modal HTML
// Inject Sure Shot Popup (Facebook Chat Style - Bottom Right)
if (!document.getElementById('sureShotModal')) {
    const modal = `
        <div id="sureShotModal" style="display:none; position:fixed; bottom:20px; right:20px; width:300px; background:#1e222d; border-radius:12px; border:1px solid #333; box-shadow:0 10px 30px rgba(0,0,0,0.5); z-index:10000; overflow:hidden; font-family:'Inter', sans-serif; animation: slideInRight 0.3s ease-out;">
            <!-- Header -->
            <div style="background:#0ecb81; padding:10px 15px; display:flex; justify-content:space-between; align-items:center;">
                <div style="font-weight:bold; color:#000; font-size:14px;">🚀 Sure Shot Entry</div>
                <div onclick="document.getElementById('sureShotModal').style.display='none'" style="cursor:pointer; color:#000; font-weight:bold; font-size:16px;">&times;</div>
            </div>
            
            <!-- Body -->
            <div style="padding:15px;">
                <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:10px;">
                    <div id="sureShotSymbol" style="font-size:18px; font-weight:bold; color:#fff;">SYMBOL</div>
                    <div id="sureShotConf" style="font-size:12px; background:#rgba(14,203,129,0.2); color:#0ecb81; padding:2px 6px; border-radius:4px;">100%</div>
                </div>
                
                <div id="sureShotReason" style="color:#aaa; font-size:12px; line-height:1.4; margin-bottom:15px;">
                    Reasoning text goes here...
                </div>
                
                <button onclick="document.getElementById('sureShotModal').style.display='none'" 
                   style="width:100%; background:#2a2e39; color:#fff; border:none; padding:8px; font-size:12px; border-radius:6px; cursor:pointer; hover:background:#333;">
                    Acknowledge
                </button>
            </div>
        </div>
        <style>
            @keyframes slideInRight { from { transform: translateX(100%); opacity: 0; } to { transform: translateX(0); opacity: 1; } }
        </style>`;
    document.body.insertAdjacentHTML('beforeend', modal);
}

let sureShotSeen = new Set();
let sureShotQueue = [];
let lastSureShotTime = 0;

// Periodic Queue Processor
setInterval(() => {
    const now = Date.now();
    const cooldown = (lastSureShotTime === 0) ? 5000 : 15000; // 15s cooldown between popups

    if (now - lastSureShotTime > cooldown && sureShotQueue.length > 0) {
        sureShotQueue.sort((a, b) => b.conf - a.conf);
        const topItem = sureShotQueue.shift();
        showSureShot(topItem);
        lastSureShotTime = now;
    }
}, 1000);

function checkSureShot(data) {
    if (!data) return;
    const list = Array.isArray(data) ? data : [data];



    list.forEach(item => {
        const signal = item.signal || item.sig;
        const conf = item.conf;
        const symbol = currentSymbol;
        const reason = item.text_en || "High Confidence Paper Signal";

        // Market Filter Logic for Paper
        const isUS = !symbol.includes('.NS') && !symbol.includes('.BO');
        const isIN = symbol.includes('.NS') || symbol.includes('.BO');

        // Use currentCurrency to determine market (USD = US, INR = IN)
        if (currentCurrency === 'INR' && isUS) return;
        if (currentCurrency === 'USD' && isIN) return;

        // 90% Threshold logic
        if (conf >= 90 && signal !== 'WAIT' && signal !== 'HOLD' && !sureShotSeen.has(symbol)) {
            // Check if already queued
            const exists = sureShotQueue.find(q => q.s === symbol);
            if (!exists) {
                sureShotQueue.push({ s: symbol, n: symbol, sig: signal, reason: reason, conf: conf });
                sureShotSeen.add(symbol);
                console.log(`Queued Paper Sure Shot: ${symbol}`);
            }
        }
    });
}

function showSureShot(item) {
    const popup = document.getElementById('sureShotModal'); // Fixed ID reference
    if (!popup) return;

    // Color Logic
    const isBuy = item.sig.includes('BUY');
    const color = isBuy ? '#0ecb81' : '#f6465d';

    // Update Popup Content
    const title = popup.querySelector('h2');
    if (title) title.style.color = "#fff";

    const symDiv = document.getElementById('sureShotSymbol');
    if (symDiv) symDiv.innerText = item.n;

    const reasonDiv = document.getElementById('sureShotReason');
    if (reasonDiv) reasonDiv.innerText = item.reason;

    popup.style.display = 'flex';

    // Auto-hide
    setTimeout(() => {
        popup.style.display = 'none';
    }, 15000);
}

// --- BACKGROUND SCANNER (Fixes "Scan entire market" for Paper) ---
function initPaperScanner() {
    console.log("🚀 Starting Paper Scanner...");
    runPaperScan();
    setInterval(runPaperScan, 60000); // Check every 60s
}

async function runPaperScan() {
    try {
        // Determine Market: 'IN' or 'US' based on Currency
        // Fallback: If currentCurrency is undefined, default to 'INR' (IN)
        const cur = (typeof currentCurrency !== 'undefined') ? currentCurrency : 'INR';
        const market = (cur === 'USD') ? 'US' : 'IN';

        console.log(`📡 Paper Scanner Running. Currency: ${cur}, Market: ${market}`);

        const res = await fetch(`../api.php?action=scan_market&market=${market}&token=${API_TOKEN}`);
        const data = await res.json();

        // FIX: API returns { intraday: [], swing: [], ... }, NOT an array directly.
        // We only care about Intraday signals for Sure Shot Popup
        let signals = [];
        if (data && data.intraday && Array.isArray(data.intraday)) {
            signals = data.intraday;
        } else if (Array.isArray(data)) {
            signals = data; // Fallback if format changes
        }

        console.log(`found ${signals.length} signals`);

        if (signals.length > 0) {
            checkSureShot(signals);
        }

        // --- NEW: POPULATE TICKERS (Ported from Main App) ---
        if (data.intraday) renderTickerRow('ticker-intraday', data.intraday);
        if (data.swing) renderTickerRow('ticker-swing', data.swing);
        if (data.longterm) renderTickerRow('ticker-longterm', data.longterm);

    } catch (e) {
        console.error("Scanner Error:", e);
    }
}

function renderTickerRow(id, data) {
    const wrap = document.querySelector(`#${id} .ticker-move`);
    if (!wrap) return;
    const priority = { 'STRONG BUY': 1, 'STRONG SELL': 2, 'BUY': 3, 'ACCUMULATE': 3, 'INVEST': 1, 'SIP MODE': 3, 'SELL': 4, 'EXIT': 2, 'WAIT': 5, 'HOLD': 5 };
    const sortedData = [...data].sort((a, b) => (priority[a.sig] || 5) - (priority[b.sig] || 5));
    let html = ""; let loopData = [...sortedData];
    while (loopData.length < 15) { loopData = [...loopData, ...sortedData]; }

    loopData.forEach(item => {
        let colorClass = 'rec-' + (item.type || 'WAIT');
        // Simple View for Paper
        html += `<div class="ticker-item" onclick="loadSymbolWrapper('${item.s}')"><div class="ticker-row-top"><span class="ticker-label">${item.n}</span><span class="ticker-rec ${colorClass}">${item.sig}</span></div><div class="ticker-reason">${item.reason || item.text_en}</div></div>`;
    });
    wrap.innerHTML = html;
}

function loadSymbolWrapper(symbol) {
    // 1. Update Global Var
    currentSymbol = symbol;

    // 2. Update Input Field if exists
    const input = document.getElementById('symbolInput');
    if (input) input.value = symbol;

    // 3. Trigger Load
    loadSymbol(true);

    // 4. Scroll to Top (Mobile Friendly)
    window.scrollTo({ top: 0, behavior: 'smooth' });
}

// Start Scanner on Load
document.addEventListener('DOMContentLoaded', () => {
    initPaperScanner();
});
