<?php
// trade_engine.php - Real Professional Data Engine V15 (Level 5 Learning Brain)
require_once __DIR__ . '/core/file_handler.php';
require_once __DIR__ . '/core/cache_engine.php';
// STRICTLY REAL DATA ONLY - NO SIMULATION

function get_store_path()
{
    return __DIR__ . '/store.json';
}

// --- LEVEL 7 INTELLIGENCE HELPERS ---

function calculate_time_decay($signal_time, $timeframe_minutes)
{
    if (!$signal_time)
        return 1.0;

    $elapsed = (time() - $signal_time) / 60; // Minutes
    $max_life = $timeframe_minutes * 10; // e.g. 5m signal lasts 50 mins max

    if ($elapsed > $max_life)
        return 0.0;

    // Decay curve: 1.0 initially, drops slowly then faster
    $decay = 1.0 - ($elapsed / $max_life);
    return max(0, $decay);
}

// --- LEVEL 5 BRAIN HELPERS ---

// --- LEVEL 5 BRAIN HELPERS ---
// Using feedback_engine.php for single source of truth
require_once __DIR__ . '/feedback_engine.php';

// Removed redundant definitions of get_strategy_weights and get_pattern_bias

function get_recent_performance_bias($symbol)
{
    $file = __DIR__ . '/trade_history.json';
    if (!file_exists($file))
        return 0;

    $history = json_decode(file_get_contents($file), true);
    if (!$history)
        return 0;

    // Filter for recent trades
    $recent = array_slice($history, -10); // Last 10 trades
    $bias = 0;
    $loss_streak = 0;

    foreach (array_reverse($history) as $trade) {
        if ($trade['status'] === 'LOSS')
            $loss_streak++;
        else
            break; // Stop counting at first non-loss
    }

    foreach ($recent as $t) {
        if ($t['status'] === 'WIN')
            $bias += 1;
        if ($t['status'] === 'LOSS')
            $bias -= 1;
    }

    return ['bias' => $bias, 'streak' => $loss_streak];
}

function auto_log_prediction($symbol, $horizon, $signal, $conf, $price, $context)
{
    $file = __DIR__ . '/trade_history.json';
    $history = [];
    if (file_exists($file)) {
        $history = json_decode(file_get_contents($file), true);
        if (!$history)
            $history = [];
    }

    // 1. Check if Open Trade Exists for Symbol+Horizon
    foreach ($history as $t) {
        if ($t['symbol'] === $symbol && $t['horizon'] === $horizon && $t['status'] === 'OPEN') {
            return; // Already open
        }
    }

    // 2. Add New Trade
    $new_trade = [
        "id" => uniqid(),
        "timestamp" => time(),
        "date" => date("Y-m-d H:i:s"),
        "symbol" => $symbol,
        "horizon" => $horizon,
        "signal" => $signal,
        "confidence" => $conf,
        "entry_price" => $price,
        "status" => "OPEN",
        "exit_price" => 0,
        "pnl" => 0,
        "context" => $context // SAVE CONTEXT FOR LEARNING
    ];

    $history[] = $new_trade;
    safe_file_write($file, json_encode($history, JSON_PRETTY_PRINT));
}

// --- MAIN ANALYSIS ENGINE ---
function run_analysis($symbol, $interval = '1m')
{
    $chart_data = fetch_yahoo_data($symbol, $interval);
    $daily_data = ($interval === '1d') ? $chart_data : fetch_yahoo_data($symbol, '1d');

    if (!$chart_data['success'] || empty($chart_data['candles'])) {
        $chart_data = [
            "success" => false,
            "price" => 0,
            "candles" => [],
            "analysis" => [
                "intraday" => ["signal" => "WAIT", "conf" => 0, "text_en" => "Data Unavailable", "text_ml" => "vivaram labhyamalla", "signals" => []],
                "swing" => ["signal" => "WAIT", "conf" => 0],
                "longterm" => ["signal" => "WAIT", "conf" => 0]
            ]
        ];
        return $chart_data;
    }

    // SYNC FIX: Correctly pass Daily Data for Swing/Longterm analysis
    $d_candles = ($daily_data['success'] && !empty($daily_data['candles'])) ? $daily_data['candles'] : $chart_data['candles'];

    $analysis = analyze_multi_horizon($chart_data['candles'], $d_candles, $chart_data['price'], $symbol);
    $chart_data['analysis'] = $analysis;

    return $chart_data;
}

// --- AI BRAIN (Hyper-Level) ---
function analyze_multi_horizon($intraday_candles, $daily_candles, $price, $symbol)
{
    $sentiment = calculate_sentiment($daily_candles, $symbol);
    $GLOBALS['latest_sentiment'] = $sentiment; // Share with Risk Guard
    $regime = detect_market_regime($daily_candles);

    $res_intra = predict_intraday_hyper($intraday_candles, $price, $sentiment, $regime, $symbol);
    $res_swing = predict_swing_hyper($daily_candles, $price, $sentiment, $regime);
    $res_long = predict_long_hyper($daily_candles, $price, $sentiment);

    return [
        "intraday" => $res_intra,
        "swing" => $res_swing,
        "longterm" => $res_long,
        "sentiment" => $sentiment,
        "regime" => $regime
    ];
}

// 1. INTRADAY HYPER ENGINE (Level 5 Adaptive)
// Updated to Level 8 (Regime Aware)
require_once __DIR__ . '/ai/market_state.php';
require_once __DIR__ . '/ai/learning_engine.php';
require_once __DIR__ . '/ai/risk_guard.php';
// NEW AI MODULES (Ultimate Upgrade)
require_once __DIR__ . '/ai/timeframe_master.php';
require_once __DIR__ . '/ai/quality_scorer.php';
require_once __DIR__ . '/ai/session_manager.php';
require_once __DIR__ . '/ai/position_sizer.php';

function predict_intraday_hyper($candles, $price, $sentiment, $regime_old, $symbol)
{
    // --- DECISION TRACE LOGGING START ---
    $decision_trace = [];
    $decision_trace[] = "🔎 Analysis Started for $symbol";

    // --- LEVEL 5: RISK GUARD (Anti-Revenge) ---
    // Check this FIRST before doing any math
    $guard = check_risk_status($symbol);
    if ($guard['status'] !== 'OK') {
        return [
            "signal" => "WAIT",
            "conf" => 0,
            "text_en" => $guard['msg'],
            "text_ml" => "റിസ്ക് ഗാർഡ്: " . $guard['msg'],
            "color" => "neutral",
            "signals" => ["guard" => ["value" => "Active", "weight" => -100, "desc" => $guard['msg']]]
        ];
    }

    if (count($candles) < 20)
        return ["signal" => "WAIT", "conf" => 0, "text_en" => "Initializing AI...", "text_ml" => "AI തുടങ്ങുന്നു...", "text_hi" => "AI शुरू हो रहा है...", "color" => "neutral", "signals" => []];

    // --- LEVEL 1: MARKET STATE AWARENESS ---
    $market_state_info = detect_market_state($candles);
    $market_state = $market_state_info['state']; // TRENDING, SIDEWAYS, VOLATILE
    $market_reason = $market_state_info['reason'];

    // --- LEVEL 3: SELF-LEARNING MEMORY ---
    // Check if we have learned how to handle this state better
    $experience_bias = get_experience_bias($market_state);

    // Load Learning Modules
    $weights = get_strategy_weights();
    $perf = get_recent_performance_bias($symbol);

    // Data Extraction
    $closes = array_column($candles, 'c');
    $volumes = array_column($candles, 'v');

    $rsi = calculate_rsi($closes, 14);
    $ema20 = calculate_ema($closes, 20);
    $ema50 = calculate_ema($closes, 50);
    $adx_val = calculate_adx($candles, 14);

    $vol_slice = array_slice($volumes, -20);
    $avg_vol = (count($vol_slice) > 0) ? array_sum($vol_slice) / count($vol_slice) : 1;
    $curr_vol = end($volumes);

    // --- PHASE 1: SIGNAL BREAKDOWN ---
    $signals = [];

    // Explicit Timeframe Weight
    $signals['timeframe'] = ["value" => "Timeframe Align", "weight" => 10, "desc" => "Evaluated on Current Chart"];

    // 1. Trend (Structure)
    $trend_val = 0;
    $w_trend = $weights['ema_trend'];
    if ($price > $ema20 && $ema20 > $ema50) {
        $signals['trend'] = ["value" => "Bullish Trend", "weight" => 30 * $w_trend, "score" => 0.9, "desc" => "Structure Bullish"];
        $trend_val = 1;
    } elseif ($price < $ema20 && $ema20 < $ema50) {
        $signals['trend'] = ["value" => "Bearish Trend", "weight" => -30 * $w_trend, "score" => 0.1, "desc" => "Structure Bearish"];
        $trend_val = -1;
    } elseif ($price > $ema20) {
        $signals['trend'] = ["value" => "Weak Bullish", "weight" => 15 * $w_trend, "score" => 0.6, "desc" => "Above EMA20"];
        $trend_val = 0.5;
    } elseif ($price < $ema20) {
        $signals['trend'] = ["value" => "Weak Bearish", "weight" => -15 * $w_trend, "score" => 0.4, "desc" => "Below EMA20"];
        $trend_val = -0.5;
    } else {
        $signals['trend'] = ["value" => "Neutral", "weight" => 0, "score" => 0.5, "desc" => "No clear trend"];
    }

    // 2. Momentum (RSI)
    $w_rsi = $weights['rsi_signal'];
    if ($rsi > 55 && $rsi < 75) {
        $signals['rsi'] = ["value" => "Bullish Mom.", "weight" => 20 * $w_rsi, "score" => 0.8, "desc" => "RSI Healthy (55-75)"];
    } elseif ($rsi < 45 && $rsi > 25) {
        $signals['rsi'] = ["value" => "Bearish Mom.", "weight" => -20 * $w_rsi, "score" => 0.2, "desc" => "RSI Weak (25-45)"];
    } elseif ($rsi >= 75) {
        $signals['rsi'] = ["value" => "Overbought", "weight" => -5 * $w_rsi, "score" => 0.6, "desc" => "RSI > 75 (Risk)"];
    } elseif ($rsi <= 25) {
        $signals['rsi'] = ["value" => "Oversold", "weight" => 5 * $w_rsi, "score" => 0.4, "desc" => "RSI < 25 (Bounce?)"];
    } else {
        $signals['rsi'] = ["value" => "Neutral", "weight" => 0, "score" => 0.5, "desc" => "RSI 45-55 (Wait)"];
    }

    // 3. Volume
    $w_vol = $weights['volume_spike'];
    if ($curr_vol > $avg_vol * 1.5) {
        $trend_dir = ($signals['trend']['weight'] > 0) ? 1 : -1;
        $signals['volume'] = ["value" => "High Volume", "weight" => (10 * $w_vol) * $trend_dir, "desc" => "Vol 1.5x Avg"];
    } elseif ($curr_vol < $avg_vol * 0.5) {
        $signals['volume'] = ["value" => "Low Volume", "weight" => -5 * $w_vol, "desc" => "Weak Activity"];
    }

    // 4. News
    $w_news = $weights['news_bias'];
    if ($sentiment > 0.2) {
        $signals['news'] = ["value" => "Positive", "weight" => 10 * $w_news, "desc" => "News Support"];
    } elseif ($sentiment < -0.2) {
        $signals['news'] = ["value" => "Negative", "weight" => -10 * $w_news, "desc" => "News Drag"];
    } else {
        $signals['news'] = ["value" => "No News", "weight" => 0, "desc" => "Neutral Environment"];
    }

    // 4.5 Trend Exhaustion (Time Decay)
    // Check how long we've been above/below EMA20
    $ema20_series = calculate_ema_series($closes, 20);
    $trend_dur = 0;
    $idx = count($closes) - 1;
    if ($trend_val > 0) { // Bullish
        while ($idx > 20 && $closes[$idx] > $ema20_series[$idx]) {
            $trend_dur++;
            $idx--;
        }
    } elseif ($trend_val < 0) { // Bearish
        while ($idx > 20 && $closes[$idx] < $ema20_series[$idx]) {
            $trend_dur++;
            $idx--;
        }
    }

    if ($trend_dur > 40 && abs($rsi - 50) < 10) { // Old trend (3+ hrs) & losing momentum
        $signals['decay'] = ["value" => "Trend Exhaustion", "weight" => -15, "desc" => "Old Trend ($trend_dur bars)"];
    } elseif ($trend_dur > 60) {
        $signals['decay'] = ["value" => "Overextended", "weight" => -10, "desc" => "Very Old Trend"];
    }

    // 5. Pattern Memory Bias (The "Deja Vu" Effect)
    $context = ['rsi' => $rsi, 'trend_val' => $trend_val, 'market_state' => $market_state];
    $pattern_bias = get_pattern_bias($context);
    if ($pattern_bias != 0) {
        $signals['memory'] = ["value" => "Pattern Match", "weight" => $pattern_bias, "desc" => "Visual Memory Match"];
    }

    // --- LEVEL 3: SELF-LEARNING ---
    if ($experience_bias != 0) {
        // Bias learned from past WIN/LOSS in this Market State
        $signals['experience'] = ["value" => "Self Learning", "weight" => $experience_bias * 10, "desc" => "Adaptive Bias ($market_state)"];
    }

    // --- PHASE 1: CONFIDENCE CALCULATION ---
    $total_weight = 0;
    foreach ($signals as $s) {
        if (isset($s['weight']))
            $total_weight += $s['weight'];

    }
    // Baseline 50, capped 0-100
    $final_conf = max(0, min(100, 50 + $total_weight));

    // --- STAGE 3: SESSION WEIGHTING ---
    // Adjust confidence based on Session
    $sess_weight = get_session_weight('trend'); // Default to trend strategy weight
    $final_conf *= $sess_weight;
    $decision_trace[] = "⏰ Session Adjustment: " . get_current_session() . " (x$sess_weight)";

    // Decision Logic
    $signal = "WAIT";
    if ($final_conf >= 75)
        $signal = "BUY";
    elseif ($final_conf <= 25)
        $signal = "SELL";

    // --- STAGE 2: DECISION ACCURACY GATES ---

    // 1. Timeframe Check (HTF vs LTF)
    // We need HTF Signal. Let's assume 'regime' (passed as $regime_old) serves as HTF bias or we predict Swing.
    // For now, using Regime as HTF Proxy.
    // Regime: TRENDING (Bull/Bear implied?), We need direction.
    // Let's use EMA20 vs EMA50 from this function as LTF, and Sentiment/Regime as HTF.
    // $markets_state is LTF state.

    // 2. Trade Quality Score
    // TrendConf approx $final_conf. VolRatio $curr_vol/$avg_vol. RR (calculated below). 
    // We calculate tentative RR first for scoring.
    $atr = calculate_atr($candles, 14);
    $tentative_sl = $atr * 1.5;
    $tentative_tp = $atr * 3.0; // 1:2
    $tentative_rr = 2.0;

    // Safe Division for Indices (Volume can be 0)
    $safe_avg_vol = ($avg_vol > 0) ? $avg_vol : 1;
    $quality_score = calculate_trade_quality($final_conf, ($curr_vol / $safe_avg_vol), $tentative_rr, 0.8);
    $decision_trace[] = "💎 Quality Score: $quality_score";

    if ($quality_score < 0.7 && $signal !== 'WAIT') {
        $signal = "WAIT";
        $signals['quality'] = ["value" => "Low Quality", "weight" => -100, "desc" => "Score $quality_score < 0.7"];
        $decision_trace[] = "❌ Blocked by Quality Filter";
    }

    // --- STAGE 1: VOLATILITY GUARD (Inside Risk Guard but checked here too if needed) ---
    // Already checked at top via check_risk_status, but can add check_volatility_safeguard($candles)
    $vol_check = check_volatility_safeguard($candles);
    if ($vol_check['status'] !== 'OK') {
        $signal = "WAIT";
        $signals['volatility_risk'] = ["value" => "Spike", "weight" => -100, "desc" => $vol_check['msg']];
        $decision_trace[] = "❌ Blocked by Volatility Spike";
    }

    // --- LEVEL 1: MARKET STATE ADAPTATION (New Brain) ---
    // Instead of simple filter, we adapt confidence based on State

    $signals['state'] = ["value" => $market_state, "weight" => 0, "desc" => $market_reason];

    if ($market_state === "SIDEWAYS") {
        // In Sideways, Trend strategies fail. We reduce their weight impact or penalize confidence.
        // We need higher threshold to breakout.
        $final_conf = ($final_conf - 50) * 0.7 + 50; // Dampen deviation from 50 (Neutral)

        $signals['adaptation'] = ["value" => "Dampening", "weight" => -10, "desc" => "Sideways: Reduced Confidence"];

        // Only trade if very strong breakout signal remains
        if ($final_conf < 70 && $final_conf > 30) {
            $signal = "WAIT";
            $signals['decision'] = ["value" => "Wait", "weight" => 0, "desc" => "Market too choppy to trade"];
        }
    } elseif ($market_state === "VOLATILE") {
        // High Risk. Reduce position sizing recommendation (later) and be stricter.
        $signals['adaptation'] = ["value" => "High Risk", "weight" => -5, "desc" => "Volatile: Caution advised"];
        // Cap confidence to prevent over-eagerness
        if ($final_conf > 90)
            $final_conf = 90;
        if ($final_conf < 10)
            $final_conf = 10;
    } elseif ($market_state === "TRENDING") {
        // Trend is friend. Boost confidence slightly if aligned.
        if ($sentiment > 0 && $final_conf > 50) {
            $final_conf += 5;
            $signals['adaptation'] = ["value" => "Boosting", "weight" => 5, "desc" => "Trending: Confidence Boost"];
        } elseif ($sentiment < 0 && $final_conf < 50) {
            $final_conf -= 5;
            $signals['adaptation'] = ["value" => "Boosting", "weight" => -5, "desc" => "Trending: Confidence Boost"];
        }
    }

    // Re-evaluate Signal after Regime Adjustment
    if ($signal !== 'WAIT') {
        if ($final_conf >= 75)
            $signal = "BUY";
        elseif ($final_conf <= 25)
            $signal = "SELL";
        else
            $signal = "WAIT";
    }

    // Setup Strength Label
    $strength_label = "Weak";
    if ($final_conf >= 80)
        $strength_label = "Strong";
    elseif ($final_conf >= 60)
        $strength_label = "Moderate-Strong";
    elseif ($final_conf >= 40)
        $strength_label = "Moderate";

    // --- PHASE 2: RISK MANAGEMENT (Level 4: Dynamic Smart SL) ---
    $atr = calculate_atr($candles, 14);

    // Default (Normal Market)
    $atr_mult_sl = 1.5;
    $atr_mult_tp = 3.0; // 1:2
    $hold_time = "10-45 mins";

    if ($market_state === 'TRENDING') {
        // Let winners run
        $atr_mult_sl = 2.0; // Wider stop to avoid noise
        $atr_mult_tp = 5.0; // 1:2.5 Aim higher
        $hold_time = "30 mins - 2 hrs";
    } elseif ($market_state === 'SIDEWAYS') {
        // Tight Scalp
        $atr_mult_sl = 1.0; // Tight stop
        $atr_mult_tp = 1.5; // Short target (1:1.5)
        $hold_time = "5-15 mins";
    } elseif ($market_state === 'VOLATILE') {
        // Survival Mode
        $atr_mult_sl = 2.5; // Very wide stop to survive wicks
        $atr_mult_tp = 3.75; // 1:1.5 conservative RR for volatility
        $hold_time = "15-60 mins";
    }

    $stoploss = 0;
    $target = 0;
    $trade_type = "Intraday " . ucfirst(strtolower($market_state));

    if ($signal == "BUY") {
        $stoploss = $price - ($atr * $atr_mult_sl);
        $target = $price + ($atr * $atr_mult_tp);
    } elseif ($signal == "SELL") {
        $stoploss = $price + ($atr * $atr_mult_sl);
        $target = $price - ($atr * $atr_mult_tp);
    }

    if ($signal != "WAIT" && $stoploss != 0) {
        $risk = abs($price - $stoploss);
        $reward = abs($target - $price);
        $rr = ($risk > 0) ? ($reward / $risk) : 0;

        if ($rr < 1.5) {
            $signal = "WAIT";
            $signals['risk'] = ["value" => "Bad RR", "weight" => -100, "desc" => "RR < 1.5 (Risk: $risk)"];
        } else {
            $signals['risk'] = ["value" => "Good RR", "weight" => 10, "desc" => "RR > 1.5"];
        }
    }

    // --- AUTO-LOGGING (FEEDBACK LOOP START) ---
    // If High Confidence (>75), log to history to learn from outcome
    if ($final_conf >= 75 && $signal !== 'WAIT') {
        auto_log_prediction($symbol, 'INTRADAY', $signal, $final_conf, $price, $context, $used_strategies);
    }

    // Logic: Invalidation Conditions
    $invalidation = [];
    if ($signal == "BUY") {
        $invalidation[] = "Price closes below EMA50 (5m)";
        $invalidation[] = "RSI drops below 40";
    } elseif ($signal == "SELL") {
        $invalidation[] = "Price closes above EMA50 (5m)";
        $invalidation[] = "RSI rises above 60";
    }

    // Levels Sanity Check
    $recent_high = max(array_column($candles, 'h'));
    $recent_low = min(array_column($candles, 'l'));
    if ($recent_low > $price * 1.5 || $recent_low < $price * 0.5) {
        $recent_low = $price * 0.95;
        $recent_high = $price * 1.05;
    }

    $final_sl = ($signal == "WAIT") ? 0 : $stoploss;
    $final_tgt = ($signal == "WAIT") ? 0 : $target;
    $color = ($signal == "BUY") ? "bullish" : (($signal == "SELL") ? "bearish" : "neutral");

    // Text Generation
    $bias_text = ($market_condition == "SIDEWAYS" && $signal != "WAIT") ? " (Range Breakout Attempt)" : "";

    // --- DETAILED NEWS-STYLE REASONING GENERATION ---

    // 1. Analyze Core Drivers
    $driver_trend = $signals['trend']['value'] ?? "Neutral";
    $driver_rsi = $signals['rsi']['value'] ?? "Neutral";
    $driver_vol = $signals['volume']['value'] ?? "Normal";
    $driver_news = $signals['news']['value'] ?? "None";

    // ENGLISH NARRATIVE
    $en = "Market is currently ";
    if ($signal === "BUY") {
        $en .= "showing bullish momentum. ";
        $en .= ($driver_trend === "Bullish Trend") ? "Trend structure is strong above EMA20. " : "Price is recovering. ";
        $en .= ($driver_rsi === "Bullish Mom.") ? "RSI indicates healthy buying interest. " : "";
        $en .= ($driver_vol === "High Volume") ? "Volume spike confirms the move. " : "Volume is moderate. ";
    } elseif ($signal === "SELL") {
        $en .= "under selling pressure. ";
        $en .= ($driver_trend === "Bearish Trend") ? "Price has broken below key EMA supports. " : "Trend is weakening. ";
        $en .= ($driver_rsi === "Bearish Mom.") ? "RSI confirms downward momentum. " : "";
        $en .= ($driver_vol === "High Volume") ? "Selling volume is high. " : "";
    } else {
        $en .= "indecisive (Wait). ";
        $en .= ($market_condition === "SIDEWAYS") ? "Market is choppy/sideways. " : "Conflicting signals detected. ";
        $en .= "Better to stay out and preserve capital.";
    }

    // MALAYALAM NARRATIVE
    $ml = "വിപണി ഇപ്പോൾ ";
    if ($signal === "BUY") {
        $ml .= "മുന്നേറ്റത്തിന്റെ സൂചന നൽകുന്നു. ";
        $ml .= ($driver_trend === "Bullish Trend") ? "EMA20-ക്ക് മുകളിൽ ശക്തമായ ട്രെൻഡ് ഉണ്ട്. " : "വില തിരിച്ചു കയറുന്നു. ";
        $ml .= ($driver_rsi === "Bullish Mom.") ? "RSI ബയിംഗ് താല്പര്യം കാണിക്കുന്നു. " : "";
        $ml .= ($driver_vol === "High Volume") ? "വോളിയം കൂടിയത് ഈ നീക്കത്തിന് ശക്തി പകരുന്നു. " : "";
    } elseif ($signal === "SELL") {
        $ml .= "നല്ല സെല്ലിംഗ് സമ്മർദ്ദത്തിലാണ്. ";
        $ml .= ($driver_trend === "Bearish Trend") ? "പ്രധാന സപ്പോർട്ടുകൾക്ക് താഴെയാണ് വില. " : "ട്രെൻഡ് ദുർബലമാകുന്നു. ";
        $ml .= ($driver_rsi === "Bearish Mom.") ? "RSI താഴേക്ക് പോകാനുള്ള സാധ്യത കാണിക്കുന്നു. " : "";
        $ml .= ($driver_vol === "High Volume") ? "സെല്ലിംഗ് വോളിയം കൂടുതലാണ്. " : "";
    } else {
        $ml .= "ഒരു വ്യക്തമായ ദിശ കാണിക്കുന്നില്ല (WAIT). ";
        $ml .= ($market_condition === "SIDEWAYS") ? "മാർക്കറ്റ് ഇപ്പോൾ 'SIDEWAYS' ആണ്. " : "വ്യത്യസ്തമായ സിഗ്നലുകൾ ഉണ്ട്. ";
        $ml .= "ഇപ്പോൾ ട്രേഡ് ചെയ്യാതിരിക്കുന്നതാണ് സുരക്ഷിതം.";
    }

    // HINDI NARRATIVE
    $hi = "बाजार अभी ";
    if ($signal === "BUY") {
        $hi .= "तेजी के संकेत दे रहा है। ";
        $hi .= ($driver_trend === "Bullish Trend") ? "EMA20 के ऊपर ट्रेंड मजबूत है। " : "कीमत में सुधार हो रहा है। ";
        $hi .= ($driver_rsi === "Bullish Mom.") ? "RSI खरीदारी में रुचि दिखा रहा है। " : "";
    } elseif ($signal === "SELL") {
        $hi .= "बिकवाली के दबाव में है। ";
        $hi .= ($driver_trend === "Bearish Trend") ? "कीमत प्रमुख सपोर्ट से नीचे है। " : "ट्रेंड कमजोर हो रहा है। ";
        $hi .= ($driver_rsi === "Bearish Mom.") ? "RSI गिरावट की ओर इशारा कर रहा है। " : "";
    } else {
        $hi .= "अनिश्चित है (WAIT)। ";
        $hi .= ($market_condition === "SIDEWAYS") ? "बाजार अभी 'SIDEWAYS' है। " : "संकेत स्पष्ट नहीं हैं। ";
        $hi .= "पूंजी सुरक्षित रखना बेहतर है।";
    }

    // --- LEVEL 6: POSITION SIZING AI (Back-End) ---
    // We assume a standard Equity (e.g. 50k) if not passed, or user sets it. 
    // Ideally we need user equity. For API signal, we return the "Scaler".
    $ai_sizing = calculate_smart_position($price, $final_sl, 100000, $final_conf); // 1 Lakh base for calculation view

    return [
        "signal" => $signal,
        "conf" => round($final_conf),
        "strength" => $strength_label,
        "text_en" => $en,
        "text_ml" => $ml,
        "text_hi" => $hi,
        "color" => $color,
        "signals" => $signals,
        "trade_type" => $trade_type,
        "hold_time" => $hold_time,
        "invalidation" => $invalidation,
        "details" => [
            "rsi" => round($rsi, 2),
            "trend" => $signals['trend']['value'],
            "ema20" => round($ema20, 2),
            "vol_ratio" => round(($curr_vol / $avg_vol), 2) . "x",
            "support" => round($final_sl, 2),
            "resistance" => round($final_tgt, 2), // Target
            "atr" => round($atr, 2),
            "entry" => round($price, 2),
            "ai_scaler" => $ai_sizing['scaler'] . "x", // Display the AI confidence scaler
            "decision_trace" => $decision_trace // LEVEL 7 Trace
        ]
    ];
}

// 2. SWING HYPER ENGINE (Professional Upgrade)
function predict_swing_hyper($candles, $price, $sentiment, $regime)
{
    if (count($candles) < 50)
        return ["signal" => "WAIT", "conf" => 0, "text_en" => "Insufficient Data", "text_ml" => "...", "text_hi" => "...", "color" => "neutral", "signals" => []];

    $closes = array_column($candles, 'c');
    $vols = array_column($candles, 'v');
    $ema20 = calculate_ema($closes, 20);
    $ema50 = calculate_ema($closes, 50);
    $rsi = calculate_rsi($closes, 14);
    $avg_vol = array_sum(array_slice($vols, -10)) / 10;
    $curr_vol = end($vols);

    // Signals
    $signals = [];
    $signals['timeframe'] = ["value" => "Daily Chart", "weight" => 0, "desc" => "Evaluated on Daily Chart"];

    // 1. Trend Structure
    if ($price > $ema50) {
        $signals['trend'] = ["value" => "Uptrend", "weight" => 30, "desc" => "Price > 50 EMA (Daily)"];
        if ($ema20 > $ema50)
            $signals['structure'] = ["value" => "Bullish Align", "weight" => 10, "desc" => "EMA 20 > 50 (Strong)"];
    } else {
        $signals['trend'] = ["value" => "Downtrend", "weight" => -30, "desc" => "Price < 50 EMA (Daily)"];
        if ($ema20 < $ema50)
            $signals['structure'] = ["value" => "Bearish Align", "weight" => -10, "desc" => "EMA 20 < 50 (Weak)"];
    }

    // 2. Momentum (RSI)
    if ($rsi > 50 && $rsi < 70)
        $signals['rsi'] = ["value" => "Bullish", "weight" => 20, "desc" => "RSI Bullish (50-70)"];
    elseif ($rsi < 50 && $rsi > 30)
        $signals['rsi'] = ["value" => "Bearish", "weight" => -20, "desc" => "RSI Bearish (30-50)"];
    elseif ($rsi >= 70)
        $signals['rsi'] = ["value" => "Overbought", "weight" => -10, "desc" => "RSI Overbought (>70)"];
    elseif ($rsi <= 30)
        $signals['rsi'] = ["value" => "Oversold", "weight" => 10, "desc" => "RSI Oversold (<30)"];

    // 3. Volume Confirmation
    if ($curr_vol > $avg_vol * 1.2) {
        $signals['volume'] = ["value" => "High", "weight" => 10, "desc" => "High Volume (+20%)"];
    } else {
        $signals['volume'] = ["value" => "Normal", "weight" => 0, "desc" => "Volume Normal"];
    }

    // 4. News Sentiment (From Cache)
    if ($sentiment > 0.2)
        $signals['news'] = ["value" => "Positive", "weight" => 10, "desc" => "News Sentiment Positive"];
    elseif ($sentiment < -0.2)
        $signals['news'] = ["value" => "Negative", "weight" => -10, "desc" => "News Sentiment Negative"];

    // 5. Trend Strength (ADX) - NEW
    $adx_val = calculate_adx($candles, 14);
    if ($adx_val > 25) {
        $signals['adx'] = ["value" => "Strong Trend", "weight" => 10, "desc" => "ADX > 25 (Trending)"];
    } else {
        $signals['adx'] = ["value" => "Weak Trend", "weight" => 0, "desc" => "ADX < 25 (Choppy)"];
    }

    // 6. Candle Pattern (Simple) - NEW
    $last_c = end($candles);
    if ($last_c['c'] > $last_c['o']) {
        $signals['candle'] = ["value" => "Green Candle", "weight" => 5, "desc" => "Last Candle Bullish"];
    } else {
        $signals['candle'] = ["value" => "Red Candle", "weight" => -5, "desc" => "Last Candle Bearish"];
    }

    // 7. Volatility Penalty (Choppy Market Protection)
    // If ATR is very low (dead market) or very high (panic), reduce confidence
    $atr = calculate_atr($candles, 14); // Recalculate or reuse if available early? Moving up.
    // ATR calculation moved down usually, but we need it here.
    // Let's assume calculate_atr works.
    $atr_pct = ($atr / $price) * 100;
    if ($atr_pct < 0.05) { // < 0.05% move is dead
        $signals['volatility'] = ["value" => "Low Volatility", "weight" => -10, "desc" => "Market too quiet"];
    } elseif ($atr_pct > 0.5) { // > 0.5% in 5m candle is panic (adjust for daily?)
        // For Daily, 0.5% is nothing. Daily ATR% is usually 1-3%.
        // Let's adjust for Daily context since this is Swing
        $signals['volatility'] = ["value" => "Normal Volatility", "weight" => 0, "desc" => "ATR Normal"];
    }

    // Confidence
    $total_weight = 0;
    foreach ($signals as $s)
        $total_weight += $s['weight'];

    // Apply Feedback Loop Adjustments (Dynamic Weighting)
    $feedback_mult = 1.0;
    // (Future: Load from feedback_engine.php to scale $total_weight)

    $final_conf = max(0, min(100, 50 + $total_weight));

    // Decision
    $signal = "WAIT";
    if ($final_conf >= 65)
        $signal = "BUY";
    elseif ($final_conf <= 35)
        $signal = "SELL";

    // Setup Strength
    $strength_label = "Weak";
    if ($final_conf >= 80)
        $strength_label = "Strong";
    elseif ($final_conf >= 60)
        $strength_label = "Moderate";

    // Risk Management (Swing = Wider Stops)
    $atr = calculate_atr($candles, 14);
    $atr_mult = 2.0; // Wider for swing

    $stoploss = 0;
    $target = 0;
    if ($signal == "BUY") {
        $stoploss = $price - ($atr * $atr_mult); // 2x ATR SL
        $target = $price + ($atr * $atr_mult * 2); // 1:2 RR
    } elseif ($signal == "SELL") {
        $stoploss = $price + ($atr * $atr_mult);
        $target = $price - ($atr * $atr_mult * 2);
    }

    $invalidation = [];
    if ($signal == "BUY")
        $invalidation[] = "Daily Close below 50 EMA";
    if ($signal == "SELL")
        $invalidation[] = "Daily Close above 50 EMA";

    // SWING NARRATIVES
    $en = "Swing View: ";
    $ml = "സ്വിംഗ് കാഴ്ചപ്പാട്: ";
    $hi = "स्विंग दृष्टिकोण: ";

    if ($signal === "BUY") {
        $en .= "Bullish structure confirmed. Price is holding above 50 EMA. Momentum is building up.";
        $ml .= "ട്രെൻഡ് ബുള്ളിഷ് ആണ് (മുകളിലേക്ക്). 50 EMA-ക്ക് മുകളിലാണ് വില. RSI നല്ല നിലയിലാണ്.";
        $hi = "तेजी का रुझान है। कीमत 50 EMA से ऊपर है और मोमेंटम मजबूत है।";
    } elseif ($signal === "SELL") {
        $en .= "Bearish structure. Price rejected from resistance. Momentum favors bears.";
        $ml .= "ട്രെൻഡ് ബെയറിഷ് ആണ് (താഴേക്ക്). വില റെസിസ്റ്റൻസിൽ തട്ടി താഴുന്നു.";
        $hi = "मंदी के संकेत हैं। कीमत प्रतिरोध से नीचे आ रही है।";
    } else {
        $en .= "Market is indecisive. Better to wait for clarity.";
        $ml .= "വിപണിയിൽ വ്യക്തതയില്ല. കാത്തിരിക്കുന്നതാണ് നല്ലത്.";
        $hi = "बाजार अनिश्चित है। स्पष्टता का इंतजार करें।";
    }

    $final_sl = ($signal == "WAIT") ? 0 : $stoploss;
    $final_tgt = ($signal == "WAIT") ? 0 : $target;

    return [
        "signal" => $signal,
        "conf" => $final_conf,
        "strength" => $strength_label,
        "trade_type" => "Swing",
        "hold_time" => "3-15 Days",
        "text_en" => $en,
        "text_ml" => $ml,
        "text_hi" => $hi,
        "color" => ($signal == "BUY") ? "bullish" : (($signal == "SELL") ? "bearish" : "neutral"),
        "signals" => $signals, // Breakdown
        "invalidation" => $invalidation,
        "details" => [
            "entry" => round($price, 2),
            "atr" => round($atr, 2),
            "support" => round($final_sl, 2),
            "resistance" => round($final_tgt, 2)
        ]
    ];
}

// 3. LONG TERM ENGINE (Professional Upgrade)
function predict_long_hyper($candles, $price, $sentiment)
{
    if (count($candles) < 200)
        return ["signal" => "WAIT", "conf" => 0, "text_en" => "Need more history", "text_ml" => "...", "color" => "neutral", "signals" => []];

    $closes = array_column($candles, 'c');
    $ema200 = calculate_ema($closes, 200);
    $ema50 = calculate_ema($closes, 50); // Golden Cross check?

    $signals = [];
    $signals['timeframe'] = ["value" => "Daily/Weekly", "weight" => 0, "desc" => "Evaluated on Long-term Trend"];

    // 1. Major Trend
    if ($price > $ema200) {
        $signals['trend'] = ["value" => "Long Bullish", "weight" => 40, "desc" => "Above 200 DMA"];
    } else {
        $signals['trend'] = ["value" => "Bearish", "weight" => -40, "desc" => "Below 200 DMA"];
    }

    // 2. Golden Cross / Death Cross
    if ($ema50 > $ema200) {
        $signals['cross'] = ["value" => "Golden Cross", "weight" => 20, "desc" => "50 > 200 EMA (Bullish)"];
    } else {
        $signals['cross'] = ["value" => "Normal", "weight" => 0, "desc" => "No Golden Cross"];
    }

    // 3. Fundamental Context (Sentiment)
    if ($sentiment > 0.3) {
        $signals['macro'] = ["value" => "Good", "weight" => 15, "desc" => "News Flow Positive"];
    }

    // 4. Price Stability
    $recent_closes = array_slice($closes, -20);
    $volatility = (max($recent_closes) - min($recent_closes)) / end($recent_closes);
    if ($volatility < 0.05) {
        $signals['stab'] = ["value" => "Stable", "weight" => 5, "desc" => "Low Volatility Consolidation"];
    }

    // 5. RSI Monthly Trend - NEW
    $rsi = calculate_rsi($closes, 14);
    if ($rsi > 60) {
        $signals['rsi'] = ["value" => "Strong Momentum", "weight" => 10, "desc" => "RSI > 60 (Bullish Zone)"];
    } elseif ($rsi < 40) {
        $signals['rsi'] = ["value" => "Weak Momentum", "weight" => -10, "desc" => "RSI < 40 (Bearish Zone)"];
    }

    // 6. Volume Accumulation - NEW
    $vols = array_column($candles, 'v');
    $avg_vol_long = array_sum($vols) / count($vols);
    $recent_vol = array_sum(array_slice($vols, -5)) / 5;
    if ($recent_vol > $avg_vol_long * 1.1) {
        $signals['accum'] = ["value" => "Accumulation", "weight" => 10, "desc" => "Vol Expanding on Trend"];
    }

    $total_weight = 0;
    foreach ($signals as $s)
        $total_weight += $s['weight'];
    $final_conf = max(0, min(100, 50 + $total_weight));

    $signal = "WAIT";
    if ($final_conf >= 70)
        $signal = "INVEST"; // Long term is BUY only usually
    elseif ($final_conf <= 30)
        $signal = "EXIT";

    $strength_label = ($final_conf > 80) ? "Strong" : "Moderate";

    // Investment Plan
    // SL is purely structural (200 DMA) or percentage (15%)
    $stoploss = 0;
    $target = 0;

    if ($signal == "INVEST") {
        $stoploss = $ema200; // Hard support
        $target = $price * 1.5; // 50% Upside base target
    }

    $final_sl = ($signal == "WAIT") ? 0 : $stoploss;
    $final_tgt = ($signal == "WAIT") ? 0 : $target;

    $invalidation = [];
    if ($signal == "INVEST")
        $invalidation[] = "Weekly Close below 200 DMA";

    return [
        "signal" => $signal,
        "conf" => $final_conf,
        "strength" => $strength_label,
        "trade_type" => "Investment",
        "hold_time" => "6-12 Months",
        "text_en" => "Long Term: " . $signal,
        "text_ml" => "ദീർഘകാല നിക്ഷേപ അവസരം: " . $signal,
        "text_hi" => "दीर्घकालिक: " . $signal,
        "color" => ($signal == "INVEST") ? "bullish" : (($signal == "EXIT") ? "bearish" : "neutral"),
        "signals" => $signals,
        "invalidation" => $invalidation,
        "details" => [
            "entry" => round($price, 2),
            "atr" => "N/A", // Not relevant for investing
            "support" => round($final_sl, 2),
            "resistance" => round($final_tgt, 2)
        ]
    ];
}

require_once __DIR__ . '/core/fetch_engine.php';

// SOURCES & HELPERS (Yahoo Finance)
function fetch_yahoo_data($symbol, $interval)
{
    // --- CACHING SYSTEM (Performance Fix) ---
    // Key: candle_RELIANCE.NS_5m
    $cache_key = "candle_" . $symbol . "_" . $interval;
    $ttl = 60; // Default 1 min for Intraday

    // Longer cache for higher timeframes
    if (in_array($interval, ['1d', '1wk', '1mo'])) {
        $ttl = 300; // 5 mins for Daily/Weekly
    }

    if (function_exists('cache_get')) {
        $cached_data = cache_get($cache_key);
        if ($cached_data) {
            // Return cached result immediately (0ms latency)
            return $cached_data;
        }
    }

    $range = '1mo'; // Fallback
    if (in_array($interval, ['1m', '2m', '5m', '15m', '30m', '90m'])) {
        $range = '5d'; // Minute data limited to ~7-60 days usually
    } elseif ($interval === '1h') {
        $range = '1y'; // 1 Year of Hourly data
    } elseif ($interval === '1d') {
        $range = '5y'; // 5 Years of Daily data
    } elseif ($interval === '1wk') {
        $range = '10y'; // 10 Years of Weekly data
    } elseif ($interval === '1mo') {
        $range = 'max';
    } elseif ($interval === '1y') {
        $range = 'max';
        $interval = '3mo'; // Fix: Yahoo rejects '1y', use '3mo' (Quarterly) as proxy
    }

    $url = "https://query1.finance.yahoo.com/v8/finance/chart/" . urlencode($symbol) . "?interval=" . $interval . "&range=" . $range;

    // Use Secure Fetch Engine (Anti-Block)
    $res = fetch_url_secure($url);

    // --- FALLBACK LOGIC (Yahoo NSE Glitch Fix) ---
    // If NSE (.NS) fails, try BSE (.BO) automatically
    if (!$res && strpos($symbol, '.NS') !== false) {
        $backup_symbol = str_replace('.NS', '.BO', $symbol);
        $url_backup = "https://query1.finance.yahoo.com/v8/finance/chart/" . urlencode($backup_symbol) . "?interval=" . $interval . "&range=" . $range;
        $res = fetch_url_secure($url_backup);
    }

    if (!$res) {
        return ["success" => false];
    }

    $json = json_decode($res, true);
    $result = $json['chart']['result'][0] ?? null;

    if ($result) {
        $meta = $result['meta'];
        $times = $result['timestamp'] ?? [];
        $quote = $result['indicators']['quote'][0] ?? [];
        $candles = [];

        $count = count($times);
        for ($i = 0; $i < $count; $i++) {
            if (isset($quote['close'][$i]) && $quote['close'][$i] !== null) { // Filter nulls
                $candles[] = [
                    't' => $times[$i],
                    'o' => $quote['open'][$i],
                    'h' => $quote['high'][$i],
                    'l' => $quote['low'][$i],
                    'c' => $quote['close'][$i],
                    'v' => $quote['volume'][$i] ?? 0
                ];
            }
        }

        // Fix: Safe Price and Change Calculation
        $price = $meta['regularMarketPrice'] ?? null;
        if (($price === null || $price == 0) && !empty($candles)) {
            $price = end($candles)['c']; // Fallback to last close
        }

        $prev_close = $meta['chartPreviousClose'] ?? $meta['previousClose'] ?? null;
        $change_pct = 0;

        if ($price !== null && $prev_close !== null && $prev_close != 0) {
            $change_pct = (($price - $prev_close) / $prev_close) * 100;
        }

        // Return Data with ALL required fields for app.js
        $final_data = [
            "success" => true,
            "price" => (float) $price,
            "change_pct" => (float) $change_pct,
            "candles" => $candles,
            "source" => "Yahoo Finance (Live)"
        ];

        // --- SAVE TO CACHE ---
        if (function_exists('cache_set')) {
            cache_set($cache_key, $final_data, $ttl);
        }

        return $final_data;
    }
    return ["success" => false];
}
function calculate_sentiment($candles, $symbol)
{
    // --- CACHING SYSTEM (Prevents Blocking) ---
    $cache_file = __DIR__ . '/news_cache.json';
    $cache = [];
    $now = time();
    $cache_duration = 1800; // 30 Minutes

    if (file_exists($cache_file)) {
        $cache = json_decode(file_get_contents($cache_file), true);
        if (isset($cache[$symbol]) && ($now - $cache[$symbol]['time'] < $cache_duration)) {
            // Return Cached Score
            return $cache[$symbol]['score'];
        }
    }

    // --- FETCH FRESH NEWS ---
    $news = fetch_google_news($symbol);

    // Default score if failed
    $final_sent = 0;

    if (!empty($news)) {
        $score = 0;
        $count = 0;

        // Keyword Dictionaries
        $positive_words = [
            'profit',
            'jump',
            'surge',
            'rise',
            'gain',
            'growth',
            'strong',
            'beat',
            'bull',
            'buy',
            'upgrade',
            'higher',
            'positive',
            'win',
            'record',
            'bonus',
            'dividend',
            'acquisition',
            'expansion',
            'approve',
            'launch',
            'contract',
            'deal',
            'partnership',
            'recover',
            'green'
        ];
        $negative_words = [
            'loss',
            'drop',
            'fall',
            'decline',
            'crash',
            'weak',
            'slump',
            'bear',
            'sell',
            'downgrade',
            'lower',
            'negative',
            'fail',
            'miss',
            'issue',
            'scam',
            'fraud',
            'investigation',
            'penalty',
            'ban',
            'lawsuit',
            'debt',
            'concern',
            'risk',
            'warn'
        ];

        foreach ($news as $item) {
            $title = strtolower($item['title'] . " " . $item['desc']);

            // Simple Scoring
            foreach ($positive_words as $w) {
                if (strpos($title, $w) !== false)
                    $score += 1.5;
            }
            foreach ($negative_words as $w) {
                if (strpos($title, $w) !== false)
                    $score -= 1.5;
            }
            $count++;
            if ($count >= 7)
                break; // Analyze max 7 headlines
        }

        // Normalize
        $final_sent = max(-1.0, min(1.0, $score / 5));
    }

    // --- UPDATE CACHE ---
    $cache[$symbol] = [
        'time' => $now,
        'score' => $final_sent
    ];
    safe_file_write($cache_file, json_encode($cache));

    return $final_sent;
}

function fetch_google_news($symbol, $lang = 'en')
{
    // Clean symbol
    $clean_sym = str_replace('.NS', '', $symbol) . " stock";

    // Config based on Language
    $hl = "en-IN";
    $gl = "IN";
    $ceid = "IN:en";

    if ($lang === 'ml') {
        $hl = "ml";
        $ceid = "IN:ml";
    } elseif ($lang === 'hi') {
        $hl = "hi";
        $ceid = "IN:hi";
    }

    // Google News RSS Feed (Dynamic Language)
    $rss_url = "https://news.google.com/rss/search?q=" . urlencode($clean_sym) . "&hl=" . $hl . "&gl=" . $gl . "&ceid=" . $ceid;

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $rss_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_ENCODING, "");
    curl_setopt($ch, CURLOPT_TIMEOUT, 6);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    $xml_str = curl_exec($ch);
    curl_close($ch);

    $news_items = [];
    if ($xml_str) {
        // Suppress warnings for malformed XML
        $xml = @simplexml_load_string($xml_str);
        if ($xml && isset($xml->channel->item)) {
            foreach ($xml->channel->item as $item) {
                // Google News descriptions sometimes contain HTML, strip it
                $desc = strip_tags((string) $item->description);
                $news_items[] = [
                    'title' => (string) $item->title,
                    'desc' => $desc,
                    'link' => (string) $item->link,
                    'date' => (string) $item->pubDate
                ];
            }
        }
    }
    return $news_items;
}

// --- LEVEL 8: ADVANCED REGIME DETECTION ---
function detect_market_regime($candles)
{
    // 1. Data Prep
    $closes = array_column($candles, 'c');
    $avg_price = end($closes);
    if ($avg_price == 0)
        return 'SIDEWAYS';

    // 2. Volatility (ATR)
    $ats = calculate_atr($candles, 14);
    $vol_pct = ($ats / $avg_price) * 100;

    // 3. Trend Strength (ADX)
    $adx = calculate_adx($candles, 14);

    // 4. Squeeze (Bollinger Bands)
    $bb = calculate_bollinger_bands($closes, 20, 2);
    $bb_width = ($bb['upper'] - $bb['lower']) / $bb['middle'] * 100;

    // --- DECISION TREE ---

    // Extreme Volatility
    if ($vol_pct > 1.5)
        return 'VOLATILE';

    // Choppy / Sideways
    if ($adx < 20)
        return 'SIDEWAYS';

    // Consolidation Squeeze
    if ($bb_width < 0.5)
        return 'CONSOLIDATION';

    // Healthy Trend
    if ($adx > 25)
        return 'TRENDING';

    return 'NORMAL';
}

function calculate_ema($data, $period)
{
    if (count($data) < $period)
        return 0;
    $k = 2 / ($period + 1);
    $ema = $data[0];
    for ($i = 1; $i < count($data); $i++)
        $ema = ($data[$i] * $k) + ($ema * (1 - $k));
    return $ema;
}

function calculate_ema_series($data, $period)
{
    $n = count($data);
    if ($n < $period)
        return array_fill(0, $n, 0);

    $ema_series = [];
    $k = 2 / ($period + 1);

    // Initial SMA
    $sum = 0;
    for ($i = 0; $i < $period; $i++)
        $sum += $data[$i];
    $ema = $sum / $period;

    // Fill first 'period' elements (approximate or 0)
    for ($i = 0; $i < $period; $i++)
        $ema_series[$i] = 0;
    $ema_series[$period - 1] = $ema;

    for ($i = $period; $i < $n; $i++) {
        $ema = ($data[$i] * $k) + ($ema * (1 - $k));
        $ema_series[$i] = $ema;
    }
    return $ema_series;
}

function calculate_rsi($data, $period = 14)
{
    if (count($data) < $period + 1)
        return 50;
    $gains = 0;
    $losses = 0;
    for ($i = 1; $i <= $period; $i++) {
        $change = $data[count($data) - $period - 1 + $i] - $data[count($data) - $period - 2 + $i];
        if ($change > 0)
            $gains += $change;
        else
            $losses += abs($change);
    }
    $avg_gain = $gains / $period;
    $avg_loss = $losses / $period;
    if ($avg_loss == 0)
        return 100;
    $rs = $avg_gain / $avg_loss;
    return 100 - (100 / (1 + $rs));
}

function calculate_atr($candles, $period = 14)
{
    if (count($candles) < $period + 1)
        return 0;
    $tr_sum = 0;
    for ($i = count($candles) - $period; $i < count($candles); $i++) {
        $h = $candles[$i]['h'];
        $l = $candles[$i]['l'];
        $cp = $candles[$i - 1]['c'];
        $tr = max($h - $l, abs($h - $cp), abs($l - $cp));
        $tr_sum += $tr;
    }
    return $tr_sum / $period;
}

function calculate_bollinger_bands($data, $period = 20, $std_dev = 2)
{
    if (count($data) < $period)
        return ["middle" => 0, "upper" => 0, "lower" => 0];

    // SMA
    $slice = array_slice($data, -$period);
    $mean = array_sum($slice) / count($slice);

    // StdDev
    $sum_sq = 0;
    foreach ($slice as $val) {
        $sum_sq += pow($val - $mean, 2);
    }
    $sd = sqrt($sum_sq / count($slice));

    return [
        "middle" => $mean,
        "upper" => $mean + ($sd * $std_dev),
        "lower" => $mean - ($sd * $std_dev)
    ];
}

function calculate_adx($candles, $period = 14)
{
    $n = count($candles);
    if ($n < ($period * 2))
        return 20; // Need history for smoothing

    $tr = [];
    $pdm = [];
    $ndm = [];

    // 1. Core calcs
    for ($i = 1; $i < $n; $i++) {
        $h = $candles[$i]['h'];
        $l = $candles[$i]['l'];
        $ph = $candles[$i - 1]['h'];
        $pl = $candles[$i - 1]['l'];
        $pc = $candles[$i - 1]['c'];

        $tr_val = max($h - $l, abs($h - $pc), abs($l - $pc));
        $up_move = $h - $ph;
        $down_move = $pl - $l;

        $pdm_val = ($up_move > $down_move && $up_move > 0) ? $up_move : 0;
        $ndm_val = ($down_move > $up_move && $down_move > 0) ? $down_move : 0;

        $tr[] = $tr_val;
        $pdm[] = $pdm_val;
        $ndm[] = $ndm_val;
    }

    // 2. Wilder Smoothing Helper
    $smooth = function ($arr, $per) {
        $ret = [];
        // First value: simple sum
        $first = array_sum(array_slice($arr, 0, $per));
        $ret[$per - 1] = $first;
        // Subsequent: Prior - (Prior/Per) + Current
        for ($j = $per; $j < count($arr); $j++) {
            $ret[$j] = $ret[$j - 1] - ($ret[$j - 1] / $per) + $arr[$j];
        }
        return $ret;
    };

    $tr_smooth = $smooth($tr, $period);
    $pdm_smooth = $smooth($pdm, $period);
    $ndm_smooth = $smooth($ndm, $period);

    // 3. DX Calculation
    $dx = [];
    for ($k = $period - 1; $k < count($tr); $k++) {
        if (!isset($tr_smooth[$k]) || $tr_smooth[$k] == 0) {
            $dx[] = 0;
            continue;
        }

        $pdi = ($pdm_smooth[$k] / $tr_smooth[$k]) * 100;
        $ndi = ($ndm_smooth[$k] / $tr_smooth[$k]) * 100;

        $div = $pdi + $ndi;
        if ($div == 0)
            $dx_val = 0;
        else
            $dx_val = abs($pdi - $ndi) / $div * 100;
        $dx[] = $dx_val;
    }

    // 4. ADX (EMA of DX)
    $final_adx = 0;
    if (count($dx) > 0) {
        // Simple smoothing for ADX: Average of last 'period' DX values
        // or true Wilder smoothing. For simplicity/speed here, simple avg of last 14.
        $recent_dx = array_slice($dx, -$period);
        $final_adx = array_sum($recent_dx) / count($recent_dx);
    }

    return $final_adx;
}
?>