<?php
// ai/learning_engine.php
// LEVEL 3: SELF-LEARNING BRAIN
// Analyzes past predictions to reinforce or penalize strategies.

function run_learning_cycle($current_prices)
{
    $history_file = __DIR__ . '/../trade_history.json';
    $memory_file = __DIR__ . '/experience_memory.json';

    if (!file_exists($history_file))
        return;
    $history = json_decode(file_get_contents($history_file), true);
    if (!$history)
        return;

    $memory = [];
    if (file_exists($memory_file)) {
        $memory = json_decode(file_get_contents($memory_file), true);
        if (!$memory)
            $memory = [];
    }

    $updated = false;

    foreach ($history as &$trade) {
        if ($trade['status'] !== 'OPEN')
            continue;

        $sym = $trade['symbol'];
        // If we have current price for this symbol
        // (In a real app, we'd pass all current prices, or fetch them)
        // Here, we assume $current_prices matches (or we skip)
        // For MVP, we skip live price check here to avoid API spam, 
        // usually this runs when data is fetched.

        // LOGIC: If we *did* know result (e.g. from a separate process), we act:
        // For now, let's assume we have a way to close trades.
        // This function is a placeholder for the "Closing Logic".

        // Let's implement 'Pattern Learning'
        // If a trade is marked WIN/LOSS externally (by user or system), we learn.
    }

    // Since we don't have a background cron job to close trades, 
    // we will implement the "Learning" part that reads "CLOSED" trades.

    // SCAN FOR NEWLY CLOSED TRADES (Unlearned)
    foreach ($history as &$trade) {
        if (($trade['status'] === 'WIN' || $trade['status'] === 'LOSS') && !isset($trade['learned'])) {
            // LEARN!
            $state = $trade['context']['market_state'] ?? 'UNKNOWN';

            if (!isset($memory[$state])) {
                $memory[$state] = ['wins' => 0, 'losses' => 0, 'trend_bias' => 0];
            }

            if ($trade['status'] === 'WIN') {
                $memory[$state]['wins']++;
                $memory[$state]['trend_bias'] += 0.1; // Boost trend bias in this state
            } else {
                $memory[$state]['losses']++;
                $memory[$state]['trend_bias'] -= 0.1; // Reduce trend bias
            }

            $trade['learned'] = true;
            $updated = true;
        }
    }

    if ($updated) {
        file_put_contents($history_file, json_encode($history, JSON_PRETTY_PRINT));
        file_put_contents($memory_file, json_encode($memory, JSON_PRETTY_PRINT));
    }
}

function get_experience_bias($market_state)
{
    $memory_file = __DIR__ . '/experience_memory.json';
    if (!file_exists($memory_file))
        return 0;

    $memory = json_decode(file_get_contents($memory_file), true);
    if (!$memory || !isset($memory[$market_state]))
        return 0;

    // Return the learned bias for this state
    return $memory[$market_state]['trend_bias'] ?? 0;
}
?>