<?php
// ai/system_guard.php
// STAGE 1: CORE SAFETY A (SURVIVAL LAYER)
// The "Circuit Breaker" for the AI.
// Runs BEFORE any strategy logic.

function run_system_guard($symbol, $candles, $sentiment)
{
    // 1. DATA SANITY CHECK
    $count = count($candles);
    if ($count < 20) {
        return ["status" => "STOP", "reason" => "Insufficient Data ($count candles)"];
    }

    // 2. VOLATILITY SPIKE (Flash Crash Guard)
    // Check last candle range vs ATR
    $last = end($candles);
    $high = $last['h'];
    $low = $last['l'];
    $close = $last['c'];
    $open = $last['o'];

    $range_pct = ($high - $low) / $close * 100;

    // Calculate simple ATR of last 5 candles for speed
    $sum_range = 0;
    for ($i = 0; $i < 5; $i++) {
        $c = $candles[$count - 1 - $i];
        $sum_range += ($c['h'] - $c['l']);
    }
    $avg_range = $sum_range / 5;
    $avg_range_pct = ($avg_range / $close) * 100;

    // RULE: If current candle is 3x larger than average -> STOP (Too volatile/Spike)
    if ($range_pct > ($avg_range_pct * 3) && $range_pct > 1.0) { // Ensure it's at least 1% move
        return ["status" => "STOP", "reason" => "Volatility Spike (Range " . round($range_pct, 2) . "%)"];
    }

    // 3. LIQUIDITY / SPREAD GUARD
    // Check Volume
    $vol = $last['v'];
    if ($vol === 0) {
        return ["status" => "STOP", "reason" => "No Liquidity (Volume 0)"];
    }

    // 4. NEWS SPIKE GUARD (Sentiment Lock)
    // If sentiment is extremely irrational (> 0.8 or < -0.8 is usually pump/dump)
    if (abs($sentiment) > 0.9) {
        return ["status" => "STOP", "reason" => "Extreme News Sentiment (Panic/Euphoria)"];
    }

    // 5. LOSS STREAK (Account Protection)
    $streak_guard = check_loss_streak($symbol);
    if ($streak_guard['status'] === 'STOP') {
        return $streak_guard;
    }

    return ["status" => "OK", "reason" => "System Healthy"];
}

function check_loss_streak($symbol)
{
    $file = __DIR__ . '/../trade_history.json';
    if (!file_exists($file))
        return ["status" => "OK"];

    $history = json_decode(file_get_contents($file), true);
    if (!$history)
        return ["status" => "OK"];

    $losses = 0;
    $consecutive = 0;
    $max_allow = 3; // Strict 3 loss limit

    // Scan last 10 trades max
    $subset = array_slice($history, -10);

    foreach (array_reverse($subset) as $t) {
        if ($t['status'] === 'LOSS') {
            $consecutive++;
        } elseif ($t['status'] === 'WIN') {
            $consecutive = 0; // Reset on win
        }

        if ($consecutive >= $max_allow) {
            // Check if recent (last 2 hours)
            // If the losses happened yesterday, we don't block today (unless we implement daily limit)
            // For now, simple continuous check.
            return ["status" => "STOP", "reason" => "Loss Streak Limit ($max_allow losses)"];
        }
    }

    return ["status" => "OK"];
}
?>