<?php
// api.php
require_once __DIR__ . '/config/auth.php';

// AUTHENTICATION CHECK
$request_token = '';
if (function_exists('getallheaders')) {
    $headers = getallheaders();
    $request_token = $headers['X-API-TOKEN'] ?? $headers['x-api-token'] ?? '';
}
if (!$request_token) {
    $request_token = $_SERVER['HTTP_X_API_TOKEN'] ?? $_GET['token'] ?? '';
}

// allow 'login' and 'logout' without token
$action = $_GET['action'] ?? '';

// Whitelist actions that depend on SESSION (Admin Panel) or LOGIN
$bypass_token = [
    'get_users',
    'add_user',
    'edit_user',
    'delete_user',
    'block_user',
    'change_own_password',
    'signup',
    'login',
    'logout',
    'assess_outcomes',
    'trigger_learning'
];

if ($request_token !== API_TOKEN && !in_array($action, $bypass_token)) {
    header('HTTP/1.0 403 Forbidden');
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized Access']);
    exit;
}

error_reporting(E_ALL);
// ini_set('display_errors', 1); // Disabled to prevent JSON corruption
ini_set('display_errors', 0);
header('Content-Type: application/json');

require_once __DIR__ . '/core/sanitize.php';
require_once __DIR__ . '/core/file_handler.php';
require_once __DIR__ . '/core/cache_engine.php';
require_once 'trade_engine.php';

// Sanitize Inputs
$action = isset($_GET['action']) ? cleanInput($_GET['action']) : '';
// Get Symbol and Interval
$symbol = isset($_GET['symbol']) ? strtoupper(cleanInput($_GET['symbol'])) : 'APOLLOHOSP.NS';
$interval = isset($_GET['interval']) ? cleanInput($_GET['interval']) : '1m';

// --- AUTHENTICATION & ADMIN ENDPOINTS ---

// --- SECURE USER MANAGEMENT (Antigravity Edition) ---

$valid_roles = ['admin', 'user'];
$valid_plans = ['free', 'premium'];

if ($action === "get_users") {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode(["status" => "error", "reason" => "Not Admin"]);
        exit;
    }

    $file = __DIR__ . '/users.json';
    if (!file_exists($file)) {
        echo json_encode([]); // File missing
        exit;
    }

    $users = json_decode(file_get_contents($file), true);
    if (!is_array($users))
        $users = [];

    // Return clean list without passwords
    $clean = [];
    foreach ($users as $u) {
        if (isset($u['password']))
            unset($u['password']);
        $clean[] = $u;
    }

    // Debug: Ensure we return a JSON array even if empty
    header('Content-Type: application/json');
    echo json_encode(array_values($clean));
    exit;
}

if ($action === "delete_user") {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode(["error" => "UA"]);
        exit;
    }

    $id = $_POST['id'];
    if ($id == 1 || $id == $_SESSION['user_id']) {
        echo json_encode(["error" => "Cannot delete yourself or Super Admin"]);
        exit;
    }

    $file = __DIR__ . '/users.json';
    $users = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    $users = array_values(array_filter($users, function ($u) use ($id) {
        return $u['id'] != $id;
    }));

    file_put_contents($file, json_encode($users, JSON_PRETTY_PRINT));
    echo json_encode(["success" => true]);
    exit;
}

if ($action === "add_user" || $action === "edit_user") {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode(["error" => "UA"]);
        exit;
    }

    $file = __DIR__ . '/users.json';
    $users = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    if (!is_array($users))
        $users = [];

    $id = $_POST['id'] ?? null;
    $username = preg_replace("/[^a-zA-Z0-9_]/", "", $_POST['username']); // Sanitize
    $password = $_POST['password'];
    $role = in_array($_POST['role'], $valid_roles) ? $_POST['role'] : 'user';
    $plan = in_array($_POST['plan'], $valid_plans) ? $_POST['plan'] : 'free';
    $status = $_POST['status'] === 'blocked' ? 'blocked' : 'active';

    if ($action === "add_user") {
        if (empty($username) || empty($password)) {
            echo json_encode(["error" => "Missing Fields"]);
            exit;
        }

        // Generate Secure ID
        $newId = 1;
        if ($users) {
            $ids = array_column($users, 'id');
            $newId = max($ids) + 1;
        }

        $newUser = [
            "id" => $newId,
            "username" => $username,
            "password" => password_hash($password, PASSWORD_DEFAULT), // Enforce Hash
            "role" => $role,
            "plan" => $plan,
            "status" => $status,
            "created_at" => date("Y-m-d H:i:s")
        ];
        $users[] = $newUser;
    } else {
        foreach ($users as &$u) {
            if ($u['id'] == $id) {
                $u['username'] = $username;
                $u['role'] = $role;
                $u['plan'] = $plan;
                $u['status'] = $status;
                if (!empty($password)) {
                    $u['password'] = password_hash($password, PASSWORD_DEFAULT); // Update Hash
                }
                break;
            }
        }
    }

    file_put_contents(__DIR__ . '/users.json', json_encode(array_values($users), JSON_PRETTY_PRINT));
    echo json_encode(["success" => true]);
    exit;
}

if ($action === "change_own_password") {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    $uid = $_SESSION['user_id'] ?? null;
    if (!$uid) {
        echo json_encode(["status" => "error", "error" => "Not Logged In"]);
        exit;
    }

    $pass = $_POST['password'] ?? '';
    if (empty($pass)) {
        echo json_encode(["status" => "error", "error" => "Password Empty"]);
        exit;
    }

    $file = __DIR__ . '/users.json';
    $users = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

    $found = false;
    foreach ($users as &$u) {
        if ($u['id'] == $uid) {
            $u['password'] = password_hash($pass, PASSWORD_DEFAULT);
            $found = true;
            break;
        }
    }

    if ($found) {
        file_put_contents($file, json_encode(array_values($users), JSON_PRETTY_PRINT));
        echo json_encode(["status" => "success", "success" => true]);
    } else {
        echo json_encode(["status" => "error", "error" => "User Not Found"]);
    }
    exit;
}

if ($action === "signup") {
    $u = preg_replace("/[^a-zA-Z0-9_]/", "", $_POST['username'] ?? '');
    $p = $_POST['password'] ?? '';

    if (empty($u) || empty($p)) {
        echo json_encode(["status" => "error", "error" => "Username and Password required"]);
        exit;
    }

    $file = __DIR__ . '/users.json';
    $users = file_exists($file) ? json_decode(file_get_contents($file), true) : [];

    // Check if user exists
    foreach ($users as $existing) {
        if (strtolower($existing['username']) === strtolower($u)) {
            echo json_encode(["status" => "error", "error" => "Username already taken"]);
            exit;
        }
    }

    // Create New User
    $newId = 1;
    if ($users) {
        $ids = array_column($users, 'id');
        $newId = max($ids) + 1;
    }

    $newUser = [
        "id" => $newId,
        "username" => $u,
        "password" => password_hash($p, PASSWORD_DEFAULT),
        "role" => "user",
        "plan" => "free",
        "status" => "active",
        "created_at" => date("Y-m-d H:i:s")
    ];

    $users[] = $newUser;
    file_put_contents($file, json_encode(array_values($users), JSON_PRETTY_PRINT));

    echo json_encode(["status" => "success", "success" => true]);
    exit;
}

if ($action === "login") {
    $u = $_POST['username'] ?? '';
    $p = $_POST['password'] ?? '';

    $file = __DIR__ . '/users.json';
    $users = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    if (!is_array($users))
        $users = [];

    foreach ($users as &$user) { // Note reference &$user for update
        if ($user['username'] === $u) {
            if ($user['status'] === 'blocked') {
                echo json_encode(["error" => "Account Blocked"]);
                exit();
            }

            // AUTHENTICATION STRATEGY (Rolling Migration)
            $verified = false;
            $needs_rehash = false;

            // 1. Check Hash
            if (password_verify($p, $user['password'])) {
                $verified = true;
                if (password_needs_rehash($user['password'], PASSWORD_DEFAULT)) {
                    $needs_rehash = true;
                }
            }
            // 2. Check Plaintext (Legacy/Reset Support)
            elseif ($user['password'] === $p) {
                $verified = true;
                $needs_rehash = true; // MUST hash plaintext
            }

            if ($verified) {
                // Auto-Upgrade Password Security
                if ($needs_rehash) {
                    $user['password'] = password_hash($p, PASSWORD_DEFAULT);
                    file_put_contents(__DIR__ . '/users.json', json_encode(array_values($users), JSON_PRETTY_PRINT));
                }

                if (session_status() === PHP_SESSION_NONE) {
                    session_start();
                }
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['role'] = $user['role'];
                $_SESSION['plan'] = $user['plan'];

                $redirect = ($user['role'] === 'admin') ? 'admin.php' : 'index.php';
                echo json_encode([
                    "status" => "success",
                    "success" => true,
                    "role" => $user['role'],
                    "redirect" => $redirect
                ]);
                exit();
            }
        }
    }
    echo json_encode(["error" => "Invalid Credentials"]);
    exit();
}

if ($action === "logout") {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    session_destroy();
    header("Location: login.php");
    exit();
}



// --- TRADING ENDPOINTS ---

try {
    if ($action === "analyze" || $action === "get_data") {
        $data = run_analysis($symbol, $interval);

        // Check Session for Premium features
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $plan = $_SESSION['plan'] ?? 'free';
        session_write_close(); // Release Lock immediately so other requests (Ticker) don't block

        if ($plan === 'free' && isset($data['analysis'])) {
            // Redact AI Data for Free Users (Strict)
            $data['analysis'] = [
                "is_locked" => true,
                "sentiment" => 0,
                "message" => "Upgrade to Premium to Unlock Analysis"
            ];
        }

        echo json_encode($data);
    }
} catch (Exception $e) {
    echo json_encode(['error' => $e->getMessage()]);
}

// --- 10. Get Trade History (Admin) ---
if ($action === 'get_trade_history') {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    $plan = $_SESSION['plan'] ?? 'free'; // Get plan from session
    session_write_close();

    if ($plan !== 'admin') {
        echo json_encode([]);
        exit;
    }
    $file = __DIR__ . '/trade_history.json';
    if (file_exists($file)) {
        header('Content-Type: application/json');
        readfile($file);
    } else {
        echo json_encode([]);
    }
    exit;
}

// --- 11. Outcome Assessor (Phase 4) ---
// --- 11. Outcome Assessor (Phase 4 - Optimized Batch) ---
// --- 11. Outcome Assessor (Phase 4 - Optimized Batch) ---
if ($action === 'assess_outcomes') {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    $role = $_SESSION['role'] ?? 'user';
    session_write_close();

    if ($role !== 'admin') {
        echo json_encode(["success" => false, "msg" => "Access Denied"]);
        exit;
    }

    $file = __DIR__ . '/trade_history.json';
    if (!file_exists($file))
        die(json_encode(["error" => "No history"]));

    $logs = json_decode(file_get_contents($file), true);
    if (!$logs)
        die(json_encode(["msg" => "Empty logs"]));

    // 1. Gather OPEN symbols
    $open_indices = [];
    $symbols_to_fetch = [];

    foreach ($logs as $i => $log) {
        if ($log['status'] === 'OPEN') {
            $open_indices[] = $i;
            if (!in_array($log['symbol'], $symbols_to_fetch)) {
                $symbols_to_fetch[] = $log['symbol'];
            }
        }
    }

    if (empty($symbols_to_fetch)) {
        header('Content-Type: application/json');
        echo json_encode(["success" => true, "updated" => 0, "msg" => "No open trades"]);
        exit;
    }

    // 2. Batch Fetch
    $batch_str = implode(',', $symbols_to_fetch);
    $url = "https://query1.finance.yahoo.com/v7/finance/quote?symbols=" . urlencode($batch_str);

    // Use Secure Fetch Engine
    require_once __DIR__ . '/core/fetch_engine.php';
    $json = fetch_url_secure($url);


    $price_map = [];
    if ($json) {
        $yf = json_decode($json, true);
        if (isset($yf['quoteResponse']['result'])) {
            foreach ($yf['quoteResponse']['result'] as $q) {
                $price_map[$q['symbol']] = $q['regularMarketPrice'];
            }
        }
    }

    // 3. Update Logs & LEARN (Modular Level 5)
    $updated = 0;
    foreach ($open_indices as $i) {
        $sym = $logs[$i]['symbol'];
        if (!isset($price_map[$sym]))
            continue;

        $curr_price = $price_map[$sym];
        $entry = $logs[$i]['entry_price'];
        $logs[$i]['exit_price'] = $curr_price;

        $diff_pct = (($curr_price - $entry) / $entry) * 100;
        $new_status = 'OPEN';

        // Result Logic
        if ($logs[$i]['signal'] === 'BUY') {
            if ($diff_pct > 0.2)
                $new_status = 'WIN';
            elseif ($diff_pct < -0.2)
                $new_status = 'LOSS';
        } elseif ($logs[$i]['signal'] === 'SELL') {
            if ($diff_pct < -0.2)
                $new_status = 'WIN';
            elseif ($diff_pct > 0.2)
                $new_status = 'LOSS';
        }

        if ($new_status !== 'OPEN') {
            $logs[$i]['status'] = $new_status;
            $logs[$i]['pnl'] = round($diff_pct, 2);
            $updated++;

            // --- LEVEL 5: EXPLICIT FEEDBACK TRIGGER ---
            // 1. Update Strategy Weights defined in feedback_engine.php
            if (function_exists('update_strategy_weights') && isset($logs[$i]['used_strategies'])) {
                update_strategy_weights($logs[$i]['used_strategies'], $new_status);
            }

            // 2. Save Pattern Memory defined in feedback_engine.php
            if (function_exists('save_pattern_memory') && isset($logs[$i]['context'])) {
                save_pattern_memory($logs[$i]['context'], $new_status);
            }
        }
    }

    // Save Logs
    safe_file_write($file, json_encode($logs, JSON_PRETTY_PRINT));

    echo json_encode(["success" => true, "updated" => $updated]);
    exit;
}

// --- 11a. Trigger Learning (Admin) ---
if ($action === 'trigger_learning') {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode(["error" => "Unauthorized"]);
        exit();
    }
    require_once 'feedback_engine.php';
    $res = run_batch_learning();
    echo json_encode($res);
    exit();
}

// --- MARKET SCANNER (LIVE TICKER) ---
if ($action === 'scan_market') {
    // 1. Determine User Role FIRST
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    $plan = $_SESSION['plan'] ?? 'free';
    session_write_close();

    // 2. Role-Based & Market-Based Caching
    $market = isset($_GET['market']) ? cleanInput($_GET['market']) : 'IN';
    $cache_key = 'scan_market_full_' . $plan . '_' . $market; // Unique cache per role AND market

    if (function_exists('cache_get')) {
        $cached_scan = cache_get($cache_key);
        if ($cached_scan) {
            echo json_encode($cached_scan);
            exit;
        }
    }

    // 4. Dynamic Market List
    $market = isset($_GET['market']) ? cleanInput($_GET['market']) : 'IN';
    $scan_list_source = [];

    if ($market === 'US') {
        $scan_list_source = [
            // --- INDICES ---
            "^GSPC",
            "^DJI",
            "^IXIC",
            "^RUT",
            "^VIX",
            "^NYA",

            // --- MEGA CAP ---
            "AAPL",
            "MSFT",
            "GOOGL",
            "GOOG",
            "AMZN",
            "NVDA",
            "META",
            "TSLA",
            "BRK-B",
            "LLY",
            "AVGO",
            "V",
            "JPM",
            "TSM",
            "WMT",
            "XOM",
            "MA",
            "UNH",
            "PG",
            "JNJ",
            "HD",
            "COST",
            "MRK",
            "ORCL",
            "ABBV",
            "CVX",
            "BAC",
            "CRM",
            "AMD",
            "PEP",
            "NFLX",
            "KO",
            "ADBE",
            "TMO",
            "DIS",
            "WFC",
            "CSCO",
            "ACN",
            "MCD",
            "LIN",
            "ABT",
            "TM",
            "DHR",
            "INTC",
            "QCOM",
            "TXN",
            "VZ",
            "NKE",
            "CMCSA",
            "PM",
            "PFE",
            "AMGN",
            "UPS",
            "INTU",
            "IBM",
            "CAT",
            "UNP",
            "LOW",
            "SPGI",
            "HON",
            "COP",
            "RTX",
            "GE",
            "AMAT",
            "GS",
            "BA",
            "PLD",
            "BLK",
            "DE",
            "BKNG",
            "T",
            "SBUX",
            "ELV",
            "MS",
            "MDT",
            "TJX",
            "ADP",
            "AXP",
            "ISRG",
            "LMT",
            "GILD",
            "MMC",
            "VRTX",
            "MDLZ",
            "ADI",
            "REGN",
            "LRCX",
            "CI",
            "SYK",
            "ETN",
            "SLB",
            "C",
            "AMT",
            "CVS",
            "MO",
            "PGR",
            "ZTS",
            "BSX",
            "CB",
            "SO",
            "FI",
            "EOG",
            "MU",
            "BDX",
            "TMUS",
            "NOC",
            "CL",
            "PANW",
            "SNPS",
            "KLAC",
            "CDNS",
            "EQIX",
            "WM",
            "CME",
            "SHW",
            "CSX",
            "ITW",
            "DUK",
            "ICE",
            "APD",

            // --- TECH & AI ---
            "PLTR",
            "AI",
            "SOUN",
            "PATH",
            "U",
            "SNOW",
            "DDOG",
            "NET",
            "CRWD",
            "ZS",
            "FTNT",
            "OKTA",
            "MDB",
            "TEAM",
            "WDAY",
            "HUBS",
            "TWLO",
            "SPLK",
            "ZM",
            "DOCU",
            "ESTC",
            "GTLB",
            "HCP",
            "BILL",
            "ASAN",
            "NCNO",

            // --- EV & MANUFACTURE ---
            "RIVN",
            "LCID",
            "NIO",
            "XPEV",
            "LI",
            "F",
            "GM",
            "STLA",
            "HMC",
            "TTM",

            // --- CRYPTO & FINTECH ---
            "COIN",
            "MARA",
            "RIOT",
            "MSTR",
            "HOOD",
            "PYPL",
            "SQ",
            "AFRM",
            "UPST",
            "SOFI",
            "NU",
            "DKNG",

            // --- RETAIL & CONSUMER ---
            "LULU",
            "NKE",
            "UAA",
            "CROX",
            "DECK",
            "ONON",
            "BURL",
            "ROST",
            "DLTR",
            "DG",
            "BJ",
            "KR",
            "TGT",
            "WMT",

            // --- ETF ---
            "SPY",
            "QQQ",
            "IWM",
            "DIA",
            "GLD",
            "SLV",
            "USO",
            "UNG",
            "TLT",
            "HYG",
            "JNK",
            "LQD",
            "XLK",
            "XLF",
            "XLV",
            "XLE",
            "XLI",
            "XLY",
            "XLP",
            "XLU",
            "XLB",
            "ARKK",
            "ARKG",
            "ARKF",
            "ARKW",
            "ARKQ",
            "ARKX",
            "SMH",
            "SOXL",
            "SOXS",
            "TQQQ",
            "SQQQ"
        ];
    } else {
        // Default IN
        $scan_list_source = [
            // --- INDICES ---
            "^NSEI",
            "^NSEBANK",
            "^CNXIT",
            "^BSESN",
            "^CNXAUTO",
            "^CNXENERGY",
            "^CNXPHARMA",
            "^CNXFMCG",
            "^CNXMETAL",
            "^CNXREALTY",

            // --- NIFTY 50 & 100 (HIGH LIQUIDITY) ---
            "RELIANCE.NS",
            "TCS.NS",
            "HDFCBANK.NS",
            "ICICIBANK.NS",
            "INFY.NS",
            "BHARTIARTL.NS",
            "ITC.NS",
            "SBIN.NS",
            "LICI.NS",
            "HINDUNILVR.NS",
            "LT.NS",
            "BAJFINANCE.NS",
            "HCLTECH.NS",
            "MARUTI.NS",
            "SUNPHARMA.NS",
            "ADANIENT.NS",
            "KOTAKBANK.NS",
            "TITAN.NS",
            "ONGC.NS",
            "TATAMOTORS.NS",
            "NTPC.NS",
            "AXISBANK.NS",
            "ADANIPORTS.NS",
            "ULTRACEMCO.NS",
            "ASIANPAINT.NS",
            "COALINDIA.NS",
            "WIPRO.NS",
            "M&M.NS",
            "BAJAJFINSV.NS",
            "BAJAJ-AUTO.NS",
            "POWERGRID.NS",
            "NESTLEIND.NS",
            "JSWSTEEL.NS",
            "ADANIPOWER.NS",
            "ADANIGREEN.NS",
            "GRASIM.NS",
            "TATASTEEL.NS",
            "ZOMATO.NS",
            "VBL.NS",
            "DLF.NS",
            "SBILIFE.NS",
            "HDFCLIFE.NS",
            "BEL.NS",
            "HAL.NS",
            "LTIM.NS",
            "SIEMENS.NS",
            "IOC.NS",
            "VARROC.NS",
            "VEDL.NS",
            "PIDILITIND.NS",
            "DIVISLAB.NS",
            "INDUSINDBK.NS",
            "EICHERMOT.NS",
            "BPCL.NS",
            "GODREJCP.NS",
            "SHREECEM.NS",
            "TRENT.NS",
            "DRREDDY.NS",
            "CIPLA.NS",
            "TECHM.NS",
            "GAIL.NS",
            "AMBUJACEM.NS",
            "BRITANNIA.NS",
            "BANKBARODA.NS",
            "INDIGO.NS",
            "TATAPOWER.NS",
            "ABB.NS",
            "HAVELLS.NS",
            "JIOFIN.NS",
            "CHOLAFIN.NS",
            "PNB.NS",
            "TVSMOTOR.NS",
            "CANBK.NS",
            "APOLLOHOSP.NS",
            "DABUR.NS",
            "JINDALSTEL.NS",
            "POLYCAB.NS",
            "NAUKRI.NS",
            "ATGL.NS",
            "ICICIPRULI.NS",
            "SRF.NS",
            "MARICO.NS",
            "SBICARD.NS",
            "HEROMOTOCO.NS",
            "MUTHOOTFIN.NS",
            "TRITURBINE.NS",
            "BOSCHLTD.NS",
            "CUMMINSIND.NS",
            "COLPAL.NS",
            "BERGEPAINT.NS",

            // --- MIDCAPS & F&O STOCKS (ACTIVE TRADING) ---
            "ABCAPITAL.NS",
            "ABFRL.NS",
            "ACC.NS",
            "ADANIENSOL.NS",
            "ALKEM.NS",
            "AMBUJACEM.NS",
            "APOLLOTYRE.NS",
            "ASHOKLEY.NS",
            "ASTRAL.NS",
            "ATUL.NS",
            "AUBANK.NS",
            "AUROPHARMA.NS",
            "BAJAJHLDNG.NS",
            "BALKRISIND.NS",
            "BALRAMCHIN.NS",
            "BANDHANBNK.NS",
            "BANKINDIA.NS",
            "BATAINDIA.NS",
            "BEL.NS",
            "BHARATFORG.NS",
            "BHEL.NS",
            "BIOCON.NS",
            "BSOFT.NS",
            "CANFINHOME.NS",
            "CHAMBLFERT.NS",
            "CHOLAFIN.NS",
            "CITYUNIONB.NS",
            "COFORGE.NS",
            "CONCOR.NS",
            "COROMANDEL.NS",
            "CROMPTON.NS",
            "CUB.NS",
            "CUMMINSIND.NS",
            "DALBHARAT.NS",
            "DEEPAKNTR.NS",
            "DELTACORP.NS",
            "DIXON.NS",
            "LALPATHLAB.NS",
            "ESCORTS.NS",
            "EXIDEIND.NS",
            "FEDERALBNK.NS",
            "GMRINFRA.NS",
            "GLENMARK.NS",
            "GODREJPROP.NS",
            "GRANULES.NS",
            "GNFC.NS",
            "GUJGASLTD.NS",
            "HDFCAMC.NS",
            "HINDCOPPER.NS",
            "HINDPETRO.NS",
            "IDFC.NS",
            "IDFCFIRSTB.NS",
            "IBULHSGFIN.NS",
            "INDIAMART.NS",
            "IEX.NS",
            "INDHOTEL.NS",
            "IGL.NS",
            "INDUSTOWER.NS",
            "INTELLECT.NS",
            "IPCALAB.NS",
            "JKCEMENT.NS",
            "JSL.NS",
            "JSWENERGY.NS",
            "JUBLFOOD.NS",
            "L&TFH.NS",
            "LICHSGFIN.NS",
            "LAURUSLAB.NS",
            "LTTS.NS",
            "LUPIN.NS",
            "MANAPPURAM.NS",
            "MFSL.NS",
            "METROPOLIS.NS",
            "MGL.NS",
            "MINDTREE.NS",
            "MPHASIS.NS",
            "MRF.NS",
            "MCX.NS",
            "NAM-INDIA.NS",
            "NATIONALUM.NS",
            "NAVINFLUOR.NS",
            "NMDC.NS",
            "OBEROIRLTY.NS",
            "OFSS.NS",
            "OIL.NS",
            "PAGEIND.NS",
            "PERSISTENT.NS",
            "PETRONET.NS",
            "PFC.NS",
            "PIIND.NS",
            "PNB.NS",
            "PVRINOX.NS",
            "RAIN.NS",
            "RAMCOCEM.NS",
            "RBLBANK.NS",
            "RECLTD.NS",
            "SAIL.NS",
            "SYNGENE.NS",
            "TATACHEM.NS",
            "TATACOMM.NS",
            "TATACONSUM.NS",
            "TORNTPOWER.NS",
            "TORNTPHARM.NS",
            "UBL.NS",
            "ULTRACEMCO.NS",
            "UPL.NS",
            "VOLTAS.NS",
            "WHIRLPOOL.NS",
            "ZEEL.NS",
            "ZYDUSLIFE.NS",

            // --- POPULAR SMALLCAPS ---
            "SUZLON.NS",
            "IDEA.NS",
            "YESBANK.NS",
            "RPOWER.NS",
            "JPPOWER.NS",
            "RENUKA.NS",
            "RVNL.NS",
            "IRFC.NS",
            "HUDCO.NS",
            "NBCC.NS",
            "SJVN.NS",
            "NHPC.NS",
            "FACT.NS",
            "MMTC.NS",
            "TRIDENT.NS",
            "URJA.NS",
            "VIKASECO.NS",
            "GTLINFRA.NS",
            "SOUTHBANK.NS",
            "UCOBANK.NS",
            "MAHABANK.NS",
            "IOB.NS",
            "CENTRALBK.NS",
            "KSL.NS",
            "BCG.NS",
            "EASEMYTRIP.NS",
            "PAYTM.NS",
            "NYKAA.NS",
            "POLICYBZR.NS",
            "DELHIVERY.NS",
            "AWL.NS",
            "LODHA.NS",
            "PATANJALI.NS",
            "EQUITASBNK.NS",
            "UJJIVANSFB.NS",
            "MASTEK.NS",
            "BSOFT.NS",
            "TANLA.NS",
            "ROUTE.NS",
            "AFFLE.NS",
            "HAPPSTMNDS.NS",
            "KPITTECH.NS",

            // --- SECTOR LEADERS ---
            "KAJARIACER.NS",
            "CERA.NS",
            "AIAENG.NS",
            "TIMKEN.NS",
            "SKFINDIA.NS",
            "SCHAEFFLER.NS",
            "SUNDRMFAST.NS",
            "ENDURANCE.NS",
            "MINDAIND.NS",
            "THERMAX.NS",
            "KEI.NS",
            "FINCABLES.NS",
            "VGUARD.NS",
            "CROMPTON.NS",
            "TTKPRESTIG.NS",
            "HAWKINS.NS",
            "SYMPHONY.NS",
            "BLUESTARCO.NS"
        ];
    }

    $full_list = $scan_list_source;

    // Trigger Auto-Learning Check
    trigger_auto_learning();

    // User Request: Scan ALL stocks (Performance warning: might be slower)
    shuffle($full_list);
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 5; // Reduced to 5 for Speed/Reliability
    $scan_list = array_slice($full_list, 0, $limit);

    $intraday = [];
    $swing = [];
    $longterm = [];

    // Plan already checked above
    // Analyze each

    // Analyze each
    foreach ($scan_list as $sym) {

        // LOCK for Free Users
        if ($plan !== 'premium' && $plan !== 'admin') {
            $intraday[] = ['s' => $sym, 'n' => str_replace('.NS', '', $sym), 'sig' => 'LOCK', 'reason' => 'Upgrade to Unlock', 'type' => 'locked'];
            $swing[] = ['s' => $sym, 'n' => str_replace('.NS', '', $sym), 'sig' => 'LOCK', 'reason' => 'Upgrade to Unlock', 'type' => 'locked'];
            $longterm[] = ['s' => $sym, 'n' => str_replace('.NS', '', $sym), 'sig' => 'LOCK', 'reason' => 'Upgrade to Unlock', 'type' => 'locked'];
            continue;
        }

        $interval = $_GET['interval'] ?? '5m'; // Use User Interval or Default 5m
        $data = run_analysis($sym, $interval);

        // Critical Fix: Keys match trade_engine.php
        if ($data && isset($data['analysis'])) {
            // Extract AI Results
            $intra = $data['analysis']['intraday'];
            $sw = $data['analysis']['swing'];
            $lt = $data['analysis']['longterm'];

            // 1. Intraday
            $intraday[] = [
                's' => $sym,
                'n' => str_replace('.NS', '', $sym),
                'sig' => $intra['signal'],
                // Force Clean Technical Text for Ticker (No News Headlines)
                'reason' => "Trend: " . ($intra['signals']['trend']['value'] ?? 'N/A') . " | RSI: " . ($intra['signals']['rsi']['value'] ?? 'N/A') . " | Vol: " . ($intra['signals']['volume']['value'] ?? 'N/A'),
                'type' => strtolower(str_replace(' ', '-', $intra['signal'])),
                'brain' => $intra['signals'], // Level 10: Brain View Data
                'conf' => $intra['conf'] // Score for progress bar
            ];

            // 2. Swing
            $swing[] = [
                's' => $sym,
                'n' => str_replace('.NS', '', $sym),
                'sig' => $sw['signal'],
                'reason' => $sw['text_en'],
                'type' => strtolower(str_replace(' ', '-', $sw['signal'])),
                'brain' => $sw['signals'], // Level 10
                'conf' => $sw['conf']
            ];

            // 3. Long Term
            $longterm[] = [
                's' => $sym,
                'n' => str_replace('.NS', '', $sym),
                'sig' => $lt['signal'],
                'reason' => $lt['text_en'],
                'type' => strtolower(str_replace(' ', '-', $lt['signal'])),
                'brain' => $lt['signals'], // Level 10
                'conf' => $lt['conf']
            ];
        }
    }

    // Fallbacks if empty (to avoid empty bar)
    if (empty($intraday))
        $intraday[] = ['s' => 'NIFTY', 'n' => 'Market', 'sig' => 'WAIT', 'reason' => 'Scanning...', 'type' => 'wait'];
    if (empty($swing))
        $swing[] = ['s' => 'NIFTY', 'n' => 'Market', 'sig' => 'WAIT', 'reason' => 'Scanning...', 'type' => 'wait'];
    if (empty($longterm))
        $longterm[] = ['s' => 'NIFTY', 'n' => 'Market', 'sig' => 'HOLD', 'reason' => 'Scanning...', 'type' => 'wait'];

    $result_payload = [
        "intraday" => $intraday,
        "swing" => $swing,
        "longterm" => $longterm
    ];

    if (function_exists('cache_set')) {
        cache_set($cache_key, $result_payload, 45); // Cache for 45 seconds
    }

    echo json_encode($result_payload);
    exit();
}
// --- 13. GENERAL MARKET NEWS ENDPOINT (NEW) ---
if ($action === 'get_market_news') {
    $lang = $_GET['lang'] ?? 'en';
    $cache_key = "market_news_" . $lang;

    if (function_exists('cache_get')) {
        $cached = cache_get($cache_key);
        if ($cached) {
            echo json_encode($cached);
            exit();
        }
    }

    // Search for broad market news in requested language
    $query = "Indian Stock Market"; // Simpler query works better for multilingual
    if ($lang === 'ml')
        $query = "ഓഹരി വിപണി വാർത്തകൾ";
    if ($lang === 'hi')
        $query = "शेयर बाजार समाचार";

    $news_items = fetch_google_news($query, $lang);

    // Sort by latest
    usort($news_items, function ($a, $b) {
        return strtotime($b['date']) - strtotime($a['date']);
    });

    echo json_encode(["news" => array_slice($news_items, 0, 10)]); // Return top 10
    exit();
}
// --- 14. UNIVERSAL SYMBOL SEARCH (Yahoo Proxy) ---
if ($action === 'search_symbol') {
    $query = $_GET['query'] ?? '';
    // Proxy Yahoo Finance Search
    $url = "https://query1.finance.yahoo.com/v1/finance/search?q=" . urlencode($query) . "&quotesCount=10&newsCount=0&enableFuzzyQuery=false";

    // Use Secure Fetch Engine
    require_once __DIR__ . '/core/fetch_engine.php';
    $res = fetch_url_secure($url);


    $json = json_decode($res, true);
    $quotes = $json['quotes'] ?? [];

    $results = [];
    foreach ($quotes as $q) {
        if (isset($q['symbol'])) {
            $results[] = [
                'symbol' => $q['symbol'],
                'name' => $q['shortname'] ?? $q['longname'] ?? $q['symbol'],
                'exch' => $q['exchange'] ?? ''
            ];
        }
    }

    echo json_encode(['results' => $results]);
    exit();
}
// --- 13. AUTO-LEARNING SYSTEM (Pseudo-Cron) ---
function trigger_auto_learning()
{
    $state_file = __DIR__ . '/core/system_state.json';
    $now = time();
    $last_run = 0;

    if (file_exists($state_file)) {
        $state = json_decode(file_get_contents($state_file), true);
        $last_run = $state['last_learning'] ?? 0;
    }

    // Checks if 1 Hour (3600s) has passed
    if ($now - $last_run > 3600) {
        require_once 'feedback_engine.php';
        if (function_exists('run_batch_learning')) {
            run_batch_learning();

            // Save state
            $new_state = ['last_learning' => $now];
            safe_file_write($state_file, json_encode($new_state));
        }
    }
}
?>