<?php
ob_start(); // Start Buffering IMMEDIATELY to catch any whitespace/warnings from includes
// api/trade.php - Market Launch Version
// 1. Session Start (Centralized)
require_once __DIR__ . '/../core/session.php';

// 2. Error Handling
require_once __DIR__ . '/../core/error_handler.php';

// 3. Guards & Core
// 3. Guards & Core
require_once __DIR__ . '/../core/api_guard.php';
require_once __DIR__ . '/../core/sanitize.php';
require_once __DIR__ . '/../core/file_handler.php';
require_once __DIR__ . '/../core/cache_engine.php';
require_once __DIR__ . '/../trade_engine.php'; // Corrected Path to Root Engine

// 4. Suppress Errors
// 4. Error Handling (Log to file, hide from output)
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../error_log');
error_reporting(E_ALL);

// 5. Input Cleaning
$action = isset($_GET['action']) ? cleanInput($_GET['action']) : '';

// 6. Dynamic Headers
// specific actions like 'logout' might need redirects, not JSON
if ($action !== 'logout') {
    // SECURITY: Clear any previous output (warnings/notices) to ensure valid JSON
    ob_end_clean();
    header('Content-Type: application/json');
}

// Sanitize Inputs
$action = isset($_GET['action']) ? cleanInput($_GET['action']) : '';
// Get Symbol and Interval
$symbol = isset($_GET['symbol']) ? strtoupper(cleanInput($_GET['symbol'])) : 'APOLLOHOSP.NS';
$interval = isset($_GET['interval']) ? cleanInput($_GET['interval']) : '1m';

// --- AUTHENTICATION & ADMIN ENDPOINTS ---

if ($action === "login") {
    $u = $_POST['username'] ?? '';
    $p = $_POST['password'] ?? '';

    $users = json_decode(file_get_contents(__DIR__ . '/../users.json'), true);
    foreach ($users as $user) {
        if ($user['username'] === $u && $user['password'] === $p) { // In production use password_verify
            if ($user['status'] === 'blocked') {
                echo json_encode(["error" => "Account Blocked"]);
                exit();
            }
            // Session already started
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['plan'] = $user['plan'];

            $redirect = ($user['role'] === 'admin') ? 'admin/dashboard.php' : 'index.php';
            echo json_encode([
                "status" => "success",
                "success" => true,
                "role" => $user['role'],
                "redirect" => $redirect
            ]);
            exit();
        }
    }
    echo json_encode(["error" => "Invalid Credentials"]);
    exit();
}

if ($action === "logout") {
    // Session already started at top
    session_destroy();
    // Cache bust
    header("Cache-Control: no-cache, no-store, must-revalidate");
    header("Pragma: no-cache");
    header("Expires: 0");
    header("Location: ../login.php");
    exit();
}

if ($action === "get_users") {
    // Only Admin
    // Session already started
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode([]);
        exit();
    }

    $users = json_decode(file_get_contents(__DIR__ . '/../users.json'), true);
    // Hide passwords
    $clean = array_map(function ($u) {
        unset($u['password']);
        return $u;
    }, $users);
    echo json_encode($clean);
    exit();
}

if ($action === "add_user" || $action === "edit_user") {
    session_start();
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode(["error" => "Unauthorized"]);
        exit();
    }

    $users = json_decode(file_get_contents(__DIR__ . '/../users.json'), true);
    $id = $_POST['id'] ?? null;
    $username = $_POST['username'];
    $role = $_POST['role'];
    $plan = $_POST['plan'];
    $status = $_POST['status'];
    $password = $_POST['password'];

    if ($action === "add_user") {
        $newId = end($users)['id'] + 1;
        $newUser = [
            "id" => $newId,
            "username" => $username,
            "password" => $password,
            "role" => $role,
            "plan" => $plan,
            "status" => $status
        ];
        $users[] = $newUser;
    } else {
        foreach ($users as &$u) {
            if ($u['id'] == $id) {
                $u['username'] = $username;
                $u['role'] = $role;
                $u['plan'] = $plan;
                $u['status'] = $status;
                if (!empty($password))
                    $u['password'] = $password;
            }
        }
    }
    safe_file_write(__DIR__ . '/../users.json', json_encode($users, JSON_PRETTY_PRINT));
    echo json_encode(["success" => true]);
    exit();
}

if ($action === "block_user") {
    session_start();
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode(["error" => "Unauthorized"]);
        exit();
    }
    $id = $_POST['id'];
    $users = json_decode(file_get_contents(__DIR__ . '/../users.json'), true);
    foreach ($users as &$u) {
        if ($u['id'] == $id) {
            $u['status'] = ($u['status'] === 'active') ? 'blocked' : 'active';
        }
    }
    safe_file_write(__DIR__ . '/../users.json', json_encode($users, JSON_PRETTY_PRINT));
    echo json_encode(["success" => true]);
    exit();
}

if ($action === "change_own_password") {
    // Session already started
    $id = $_SESSION['user_id'];
    $pass = $_POST['password'];
    $users = json_decode(file_get_contents(__DIR__ . '/../users.json'), true);
    foreach ($users as &$u) {
        if ($u['id'] == $id) {
            $u['password'] = $pass;
        }
    }
    safe_file_write(__DIR__ . '/../users.json', json_encode($users, JSON_PRETTY_PRINT));
    echo json_encode(["success" => true]);
    exit();
}

// --- TRADING ENDPOINTS ---

try {
    if ($action === "analyze" || $action === "get_data") {
        $data = run_analysis($symbol, $interval);

        // Check Session for Premium features
        // Session already started at top
        $plan = $_SESSION['plan'] ?? 'free';
        session_write_close(); // Release Lock immediately so other requests (Ticker) don't block

        if ($plan === 'free' && isset($data['analysis'])) {
            // Redact AI Data for Free Users (Strict)
            $data['analysis'] = [
                "is_locked" => true,
                "sentiment" => 0,
                "message" => "Upgrade to Premium to Unlock Analysis"
            ];
        }

        echo json_encode($data);
    }
} catch (Exception $e) {
    echo json_encode(['error' => $e->getMessage()]);
}

// --- 10. Get Trade History (Admin) ---
if ($action === 'get_trade_history') {
    // Session already started
    $plan = $_SESSION['plan'] ?? 'free'; // Get plan from session
    session_write_close();

    if ($plan !== 'admin') {
        echo json_encode([]);
        exit;
    }
    $file = __DIR__ . '/../trade_history.json';
    if (file_exists($file)) {
        header('Content-Type: application/json');
        readfile($file);
    } else {
        echo json_encode([]);
    }
    exit;
}

// --- 11. Outcome Assessor (Phase 4) ---
if ($action === 'assess_outcomes') {
    session_start();
    $role = $_SESSION['role'] ?? 'user';
    session_write_close();

    if ($role !== 'admin') {
        echo json_encode(["success" => false, "msg" => "Access Denied"]);
        exit;
    }

    $file = __DIR__ . '/../trade_history.json';
    if (!file_exists($file))
        die(json_encode(["error" => "No history"]));

    $logs = json_decode(file_get_contents($file), true);
    if (!$logs)
        die(json_encode(["msg" => "Empty logs"]));

    // 1. Gather OPEN symbols
    $open_indices = [];
    $symbols_to_fetch = [];

    foreach ($logs as $i => $log) {
        if ($log['status'] === 'OPEN') {
            $open_indices[] = $i;
            if (!in_array($log['symbol'], $symbols_to_fetch)) {
                $symbols_to_fetch[] = $log['symbol'];
            }
        }
    }

    if (empty($symbols_to_fetch)) {
        header('Content-Type: application/json');
        echo json_encode(["success" => true, "updated" => 0, "msg" => "No open trades"]);
        exit;
    }

    // 2. Batch Fetch
    $batch_str = implode(',', $symbols_to_fetch);
    $url = "https://query1.finance.yahoo.com/v7/finance/quote?symbols=" . urlencode($batch_str);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    $json = curl_exec($ch);
    curl_close($ch);

    $price_map = [];
    if ($json) {
        $yf = json_decode($json, true);
        if (isset($yf['quoteResponse']['result'])) {
            foreach ($yf['quoteResponse']['result'] as $q) {
                $price_map[$q['symbol']] = $q['regularMarketPrice'];
            }
        }
    }

    // 3. Update Logs & LEARN (Modular Level 5)
    $updated = 0;
    foreach ($open_indices as $i) {
        $sym = $logs[$i]['symbol'];
        if (!isset($price_map[$sym]))
            continue;

        $curr_price = $price_map[$sym];
        $entry = $logs[$i]['entry_price'];
        $logs[$i]['exit_price'] = $curr_price;

        $diff_pct = (($curr_price - $entry) / $entry) * 100;
        $new_status = 'OPEN';

        // Result Logic
        if ($logs[$i]['signal'] === 'BUY') {
            if ($diff_pct > 0.2)
                $new_status = 'WIN';
            elseif ($diff_pct < -0.2)
                $new_status = 'LOSS';
        } elseif ($logs[$i]['signal'] === 'SELL') {
            if ($diff_pct < -0.2)
                $new_status = 'WIN';
            elseif ($diff_pct > 0.2)
                $new_status = 'LOSS';
        }

        if ($new_status !== 'OPEN') {
            $logs[$i]['status'] = $new_status;
            $logs[$i]['pnl'] = round($diff_pct, 2);
            $updated++;

            // --- LEVEL 5: EXPLICIT FEEDBACK TRIGGER ---
            // 1. Update Strategy Weights defined in feedback.php
            if (function_exists('update_strategy_weights') && isset($logs[$i]['used_strategies'])) {
                update_strategy_weights($logs[$i]['used_strategies'], $new_status);
            }

            // 2. Save Pattern Memory defined in feedback.php
            if (function_exists('save_pattern_memory') && isset($logs[$i]['context'])) {
                save_pattern_memory($logs[$i]['context'], $new_status);
            }
        }
    }

    // Save Logs
    safe_file_write($file, json_encode($logs, JSON_PRETTY_PRINT));

    echo json_encode(["success" => true, "updated" => $updated]);
    exit;
}

// --- 11a. Trigger Learning (Admin) ---
if ($action === 'trigger_learning') {
    session_start();
    if (($_SESSION['role'] ?? '') !== 'admin') {
        echo json_encode(["error" => "Unauthorized"]);
        exit();
    }
    require_once __DIR__ . '/../core/feedback.php';
    $res = run_batch_learning();
    echo json_encode($res);
    exit();
}

// --- MARKET SCANNER (LIVE TICKER) ---
if ($action === 'scan_market') {
    // 1. Determine User Role FIRST
    session_start();
    $plan = $_SESSION['plan'] ?? 'free';
    session_write_close();

    // 2. Role-Based Caching
    $cache_key = 'scan_market_full_' . $plan; // Unique cache per role

    if (function_exists('cache_get')) {
        $cached_scan = cache_get($cache_key);
        if ($cached_scan) {
            echo json_encode($cached_scan);
            exit;
        }
    }

    $full_list = [
        // --- INDICES ---
        "^NSEI",
        "^NSEBANK",
        "^CNXIT",
        "^BSESN",
        "^CNXAUTO",
        "^CNXENERGY",
        "^CNXPHARMA",
        "^CNXFMCG",
        "^CNXMETAL",
        "^CNXREALTY",

        // --- NIFTY 50 & 100 (HIGH LIQUIDITY) ---
        "RELIANCE.NS",
        "TCS.NS",
        "HDFCBANK.NS",
        "ICICIBANK.NS",
        "INFY.NS",
        "BHARTIARTL.NS",
        "ITC.NS",
        "SBIN.NS",
        "LICI.NS",
        "HINDUNILVR.NS",
        "LT.NS",
        "BAJFINANCE.NS",
        "HCLTECH.NS",
        "MARUTI.NS",
        "SUNPHARMA.NS",
        "ADANIENT.NS",
        "KOTAKBANK.NS",
        "TITAN.NS",
        "ONGC.NS",
        "TATAMOTORS.NS",
        "NTPC.NS",
        "AXISBANK.NS",
        "ADANIPORTS.NS",
        "ULTRACEMCO.NS",
        "ASIANPAINT.NS",
        "COALINDIA.NS",
        "WIPRO.NS",
        "M&M.NS",
        "BAJAJFINSV.NS",
        "BAJAJ-AUTO.NS",
        "POWERGRID.NS",
        "NESTLEIND.NS",
        "JSWSTEEL.NS",
        "ADANIPOWER.NS",
        "ADANIGREEN.NS",
        "GRASIM.NS",
        "TATASTEEL.NS",
        "ZOMATO.NS",
        "VBL.NS",
        "DLF.NS",
        "SBILIFE.NS",
        "HDFCLIFE.NS",
        "BEL.NS",
        "HAL.NS",
        "LTIM.NS",
        "SIEMENS.NS",
        "IOC.NS",
        "VARROC.NS",
        "VEDL.NS",
        "PIDILITIND.NS",
        "DIVISLAB.NS",
        "INDUSINDBK.NS",
        "EICHERMOT.NS",
        "BPCL.NS",
        "GODREJCP.NS",
        "SHREECEM.NS",
        "TRENT.NS",
        "DRREDDY.NS",
        "CIPLA.NS",
        "TECHM.NS",
        "GAIL.NS",
        "AMBUJACEM.NS",
        "BRITANNIA.NS",
        "BANKBARODA.NS",
        "INDIGO.NS",
        "TATAPOWER.NS",
        "ABB.NS",
        "HAVELLS.NS",
        "JIOFIN.NS",
        "CHOLAFIN.NS",
        "PNB.NS",
        "TVSMOTOR.NS",
        "CANBK.NS",
        "APOLLOHOSP.NS",
        "DABUR.NS",
        "JINDALSTEL.NS",
        "POLYCAB.NS",
        "NAUKRI.NS",
        "ATGL.NS",
        "ICICIPRULI.NS",
        "SRF.NS",
        "MARICO.NS",
        "SBICARD.NS",
        "HEROMOTOCO.NS",
        "MUTHOOTFIN.NS",
        "TRITURBINE.NS",
        "BOSCHLTD.NS",
        "CUMMINSIND.NS",
        "COLPAL.NS",
        "BERGEPAINT.NS",

        // --- MIDCAPS & F&O STOCKS (ACTIVE TRADING) ---
        "ABCAPITAL.NS",
        "ABFRL.NS",
        "ACC.NS",
        "ADANIENSOL.NS",
        "ALKEM.NS",
        "AMBUJACEM.NS",
        "APOLLOTYRE.NS",
        "ASHOKLEY.NS",
        "ASTRAL.NS",
        "ATUL.NS",
        "AUBANK.NS",
        "AUROPHARMA.NS",
        "BAJAJHLDNG.NS",
        "BALKRISIND.NS",
        "BALRAMCHIN.NS",
        "BANDHANBNK.NS",
        "BANKINDIA.NS",
        "BATAINDIA.NS",
        "BEL.NS",
        "BHARATFORG.NS",
        "BHEL.NS",
        "BIOCON.NS",
        "BSOFT.NS",
        "CANFINHOME.NS",
        "CHAMBLFERT.NS",
        "CHOLAFIN.NS",
        "CITYUNIONB.NS",
        "COFORGE.NS",
        "CONCOR.NS",
        "COROMANDEL.NS",
        "CROMPTON.NS",
        "CUB.NS",
        "CUMMINSIND.NS",
        "DALBHARAT.NS",
        "DEEPAKNTR.NS",
        "DELTACORP.NS",
        "DIXON.NS",
        "LALPATHLAB.NS",
        "ESCORTS.NS",
        "EXIDEIND.NS",
        "FEDERALBNK.NS",
        "GMRINFRA.NS",
        "GLENMARK.NS",
        "GODREJPROP.NS",
        "GRANULES.NS",
        "GNFC.NS",
        "GUJGASLTD.NS",
        "HDFCAMC.NS",
        "HINDCOPPER.NS",
        "HINDPETRO.NS",
        "IDFC.NS",
        "IDFCFIRSTB.NS",
        "IBULHSGFIN.NS",
        "INDIAMART.NS",
        "IEX.NS",
        "INDHOTEL.NS",
        "IGL.NS",
        "INDUSTOWER.NS",
        "INTELLECT.NS",
        "IPCALAB.NS",
        "JKCEMENT.NS",
        "JSL.NS",
        "JSWENERGY.NS",
        "JUBLFOOD.NS",
        "L&TFH.NS",
        "LICHSGFIN.NS",
        "LAURUSLAB.NS",
        "LTTS.NS",
        "LUPIN.NS",
        "MANAPPURAM.NS",
        "MFSL.NS",
        "METROPOLIS.NS",
        "MGL.NS",
        "MINDTREE.NS",
        "MPHASIS.NS",
        "MRF.NS",
        "MCX.NS",
        "NAM-INDIA.NS",
        "NATIONALUM.NS",
        "NAVINFLUOR.NS",
        "NMDC.NS",
        "OBEROIRLTY.NS",
        "OFSS.NS",
        "OIL.NS",
        "PAGEIND.NS",
        "PERSISTENT.NS",
        "PETRONET.NS",
        "PFC.NS",
        "PIIND.NS",
        "PNB.NS",
        "PVRINOX.NS",
        "RAIN.NS",
        "RAMCOCEM.NS",
        "RBLBANK.NS",
        "RECLTD.NS",
        "SAIL.NS",
        "SYNGENE.NS",
        "TATACHEM.NS",
        "TATACOMM.NS",
        "TATACONSUM.NS",
        "TORNTPOWER.NS",
        "TORNTPHARM.NS",
        "UBL.NS",
        "ULTRACEMCO.NS",
        "UPL.NS",
        "VOLTAS.NS",
        "WHIRLPOOL.NS",
        "ZEEL.NS",
        "ZYDUSLIFE.NS",

        // --- POPULAR SMALLCAPS ---
        "SUZLON.NS",
        "IDEA.NS",
        "YESBANK.NS",
        "RPOWER.NS",
        "JPPOWER.NS",
        "RENUKA.NS",
        "RVNL.NS",
        "IRFC.NS",
        "HUDCO.NS",
        "NBCC.NS",
        "SJVN.NS",
        "NHPC.NS",
        "FACT.NS",
        "MMTC.NS",
        "TRIDENT.NS",
        "URJA.NS",
        "VIKASECO.NS",
        "GTLINFRA.NS",
        "SOUTHBANK.NS",
        "UCOBANK.NS",
        "MAHABANK.NS",
        "IOB.NS",
        "CENTRALBK.NS",
        "KSL.NS",
        "BCG.NS",
        "EASEMYTRIP.NS",
        "PAYTM.NS",
        "NYKAA.NS",
        "POLICYBZR.NS",
        "DELHIVERY.NS",
        "AWL.NS",
        "LODHA.NS",
        "PATANJALI.NS",
        "EQUITASBNK.NS",
        "UJJIVANSFB.NS",
        "MASTEK.NS",
        "BSOFT.NS",
        "TANLA.NS",
        "ROUTE.NS",
        "AFFLE.NS",
        "HAPPSTMNDS.NS",
        "KPITTECH.NS",

        // --- SECTOR LEADERS ---
        "KAJARIACER.NS",
        "CERA.NS",
        "AIAENG.NS",
        "TIMKEN.NS",
        "SKFINDIA.NS",
        "SCHAEFFLER.NS",
        "SUNDRMFAST.NS",
        "ENDURANCE.NS",
        "MINDAIND.NS",
        "THERMAX.NS",
        "KEI.NS",
        "FINCABLES.NS",
        "VGUARD.NS",
        "CROMPTON.NS",
        "TTKPRESTIG.NS",
        "HAWKINS.NS",
        "SYMPHONY.NS",
        "BLUESTARCO.NS"
    ];

    // Trigger Auto-Learning Check
    trigger_auto_learning();

    // User Request: Scan ALL stocks (Performance warning: might be slower)
    shuffle($full_list);
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 5; // Default to 5 for speed
    $scan_list = array_slice($full_list, 0, $limit);

    $intraday = [];
    $swing = [];
    $longterm = [];

    // Analyze each
    foreach ($scan_list as $sym) {

        // LOCK for Free Users
        if ($plan !== 'premium' && $plan !== 'admin') {
            $intraday[] = ['s' => $sym, 'n' => str_replace('.NS', '', $sym), 'sig' => 'LOCK', 'reason' => 'Upgrade to Unlock', 'type' => 'locked'];
            $swing[] = ['s' => $sym, 'n' => str_replace('.NS', '', $sym), 'sig' => 'LOCK', 'reason' => 'Upgrade to Unlock', 'type' => 'locked'];
            $longterm[] = ['s' => $sym, 'n' => str_replace('.NS', '', $sym), 'sig' => 'LOCK', 'reason' => 'Upgrade to Unlock', 'type' => 'locked'];
            continue;
        }

        $data = run_analysis($sym, '5m'); // Use 5m for proper Intraday analysis in Ticker

        // Critical Fix: Keys match trade_engine.php
        if ($data && isset($data['analysis'])) {
            // Extract AI Results
            $intra = $data['analysis']['intraday'];
            $sw = $data['analysis']['swing'];
            $lt = $data['analysis']['longterm'];

            // 1. Intraday
            $intraday[] = [
                's' => $sym,
                'n' => str_replace('.NS', '', $sym),
                'sig' => $intra['signal'],
                // Force Clean Technical Text for Ticker (No News Headlines)
                'reason' => "Trend: " . ($intra['signals']['trend']['value'] ?? 'N/A') . " | RSI: " . ($intra['signals']['rsi']['value'] ?? 'N/A') . " | Vol: " . ($intra['signals']['volume']['value'] ?? 'N/A'),
                'type' => strtolower(str_replace(' ', '-', $intra['signal'])),
                'brain' => $intra['signals'], // Level 10: Brain View Data
                'conf' => $intra['conf'] // Score for progress bar
            ];

            // 2. Swing
            $swing[] = [
                's' => $sym,
                'n' => str_replace('.NS', '', $sym),
                'sig' => $sw['signal'],
                'reason' => $sw['text_en'],
                'type' => strtolower(str_replace(' ', '-', $sw['signal'])),
                'brain' => $sw['signals'], // Level 10
                'conf' => $sw['conf']
            ];

            // 3. Long Term
            $longterm[] = [
                's' => $sym,
                'n' => str_replace('.NS', '', $sym),
                'sig' => $lt['signal'],
                'reason' => $lt['text_en'],
                'type' => strtolower(str_replace(' ', '-', $lt['signal'])),
                'brain' => $lt['signals'], // Level 10
                'conf' => $lt['conf']
            ];
        }
    }

    // Fallbacks if empty (to avoid empty bar)
    if (empty($intraday))
        $intraday[] = ['s' => 'NIFTY', 'n' => 'Market', 'sig' => 'WAIT', 'reason' => 'Scanning...', 'type' => 'wait'];
    if (empty($swing))
        $swing[] = ['s' => 'NIFTY', 'n' => 'Market', 'sig' => 'WAIT', 'reason' => 'Scanning...', 'type' => 'wait'];
    if (empty($longterm))
        $longterm[] = ['s' => 'NIFTY', 'n' => 'Market', 'sig' => 'HOLD', 'reason' => 'Scanning...', 'type' => 'wait'];

    $result_payload = [
        "intraday" => $intraday,
        "swing" => $swing,
        "longterm" => $longterm
    ];

    if (function_exists('cache_set')) {
        cache_set($cache_key, $result_payload, 15); // Cache for 15 seconds
    }

    echo json_encode($result_payload);
    exit();
}
// --- 13. GENERAL MARKET NEWS ENDPOINT (NEW) ---
if ($action === 'get_market_news') {
    // Search for broad market news
    $news_items = fetch_google_news("Indian Stock Market Nifty 50");

    // Sort by latest
    usort($news_items, function ($a, $b) {
        return strtotime($b['date']) - strtotime($a['date']);
    });

    echo json_encode(["news" => array_slice($news_items, 0, 10)]); // Return top 10
    exit();
}
// --- 13. AUTO-LEARNING SYSTEM (Pseudo-Cron) ---
function trigger_auto_learning()
{
    $state_file = __DIR__ . '/../core/system_state.json';
    $now = time();
    $last_run = 0;

    if (file_exists($state_file)) {
        $state = json_decode(file_get_contents($state_file), true);
        $last_run = $state['last_learning'] ?? 0;
    }

    // Checks if 1 Hour (3600s) has passed
    if ($now - $last_run > 3600) {
        require_once __DIR__ . '/../core/feedback.php';
        if (function_exists('run_batch_learning')) {
            run_batch_learning();

            // Save state
            $new_state = ['last_learning' => $now];
            safe_file_write($state_file, json_encode($new_state));
        }
    }
}
