<?php
// feedback.php
require_once __DIR__ . '/file_handler.php';
// Level 7.2 Self-Learning Core
// Handles Strategy Weights, Pattern Memory, and Feedback Loops

// --- 1. STRATEGY WEIGHTS SYSTEM ---

function get_strategy_weights()
{
    $file = __DIR__ . '/../strategy_weights.json';
    $defaults = [
        "ema_trend" => 1.0,
        "rsi_signal" => 1.0,
        "volume_spike" => 1.0,
        "news_bias" => 1.0,
        "volatility_filter" => 1.0
    ];

    if (file_exists($file)) {
        $stored = json_decode(file_get_contents($file), true);
        if ($stored)
            return array_merge($defaults, $stored);
    }
    return $defaults;
}

function update_strategy_weights($used_strategies, $result)
{
    if (empty($used_strategies))
        return;

    $file = __DIR__ . '/../strategy_weights.json';
    $weights = get_strategy_weights();

    // Learning Rate (Adaptive)
    $delta = ($result === 'WIN') ? 0.05 : -0.05;

    foreach ($used_strategies as $strat) {
        if (isset($weights[$strat])) {

            // --- AUTO PRUNING (Stage 3) ---
            // Track consecutive losses for this strategy
            // Using a separate key in JSON for tracking streaks? 
            // For simplicity, we assume if weight hits MIN (0.1), it's dead.

            $weights[$strat] += $delta;

            // Pruning Logic: If Weight drops below 0.2, disable it (Set to 0)
            // It needs manual revive or slow heal?
            // "Strategy = 20 continuous losses" -> effectively drops weight by 1.0 (20 * 0.05).
            // Default 1.0 - 1.0 = 0. So 0.1 threshold is good.

            if ($weights[$strat] < 0.2) {
                $weights[$strat] = 0; // Disabled
                // Log this event?
            }

            // Safety Bounds (0.0 to 3.0) - Allow 0 for disabled
            $weights[$strat] = max(0.0, min(3.0, $weights[$strat]));
        }
    }

    safe_file_write($file, json_encode($weights, JSON_PRETTY_PRINT));
}

// --- 2. PATTERN MEMORY SYSTEM ---

function save_pattern_memory($pattern_context, $result)
{
    $file = __DIR__ . '/../pattern_memory.json';
    $patterns = file_exists($file) ? json_decode(file_get_contents($file), true) : [];
    if (!$patterns)
        $patterns = [];

    // Construct Pattern Object
    $new_memory = [
        "time" => time(),
        "rsi" => $pattern_context['rsi'],
        "trend_val" => $pattern_context['trend_val'], // 1 (Bull), -1 (Bear), 0 (Neutral)
        "volatility" => $pattern_context['volatility'] ?? 0,
        "result" => $result
    ];

    $patterns[] = $new_memory;

    // Memory Limit (Keep last 500 experiences)
    if (count($patterns) > 500) {
        $patterns = array_slice($patterns, -500);
    }

    safe_file_write($file, json_encode($patterns, JSON_PRETTY_PRINT));
}

function get_pattern_bias($current_context)
{
    $file = __DIR__ . '/../pattern_memory.json';
    if (!file_exists($file))
        return 0;

    $patterns = json_decode(file_get_contents($file), true);
    if (!$patterns)
        return 0;

    $bias = 0;

    foreach ($patterns as $p) {
        // Similarity Check (Euclidean-ish)
        $rsi_diff = abs($p['rsi'] - $current_context['rsi']);
        $trend_match = ($p['trend_val'] == $current_context['trend_val']);

        // Strict Match Logic
        if ($rsi_diff < 5 && $trend_match) {
            // Apply learned experience
            if ($p['result'] === 'WIN')
                $bias += 0.5; // Boost confidence
            if ($p['result'] === 'LOSS')
                $bias -= 0.5; // Reduce confidence
        }
    }

    // Limit absolute bias impact (+/- 15% max)
    return max(-15, min(15, $bias));
}

// --- 3. LOSS STREAK PROTECTION (Safety Brain) ---

function get_loss_streak_guard()
{
    $file = __DIR__ . '/../trade_history.json';
    if (!file_exists($file))
        return false;

    $history = json_decode(file_get_contents($file), true);
    if (!$history)
        return false;

    $loss_streak = 0;
    foreach (array_reverse($history) as $trade) {
        if ($trade['status'] === 'LOSS')
            $loss_streak++;
        else
            break;
    }

    // If 2 or more consecutive losses, activate guard
    return ($loss_streak >= 2);
}

// --- 4. BATCH LEARNING TRIGGER (Admin / Cron) ---

function run_batch_learning()
{
    $file = __DIR__ . '/../trade_history.json';
    if (!file_exists($file))
        return ["status" => "No History"];

    $history = json_decode(file_get_contents($file), true);
    if (!$history)
        return ["status" => "Empty History"];

    $learned_count = 0;

    // Process last 50 potential "unlearned" trades
    $recent_history = array_slice($history, -50);

    foreach ($recent_history as $trade) {
        if (!isset($trade['status']) || $trade['status'] === 'OPEN')
            continue;

        $result = $trade['status']; // WIN or LOSS
        $strategies = $trade['strategies'] ?? ['ema_trend', 'rsi_signal'];

        update_strategy_weights($strategies, $result);

        // Also update pattern memory if context exists
        if (isset($trade['context'])) {
            save_pattern_memory($trade['context'], $result);
        }
        $learned_count++;
    }

    return ["status" => "Success", "learned_from" => $learned_count];
}