<?php
// core/json_handler.php
// STAGE 6: STABILITY
// Handles JSON read/write with File Locking (flock) to prevent race conditions.

function safe_read_json($filepath, $default = [])
{
    if (!file_exists($filepath))
        return $default;

    $fp = fopen($filepath, 'r');
    if (!$fp)
        return $default;

    // Shared Lock (Reader)
    if (flock($fp, LOCK_SH)) {
        $filesize = filesize($filepath);
        if ($filesize > 0) {
            $content = fread($fp, $filesize);
            $data = json_decode($content, true);
        } else {
            $data = $default;
        }
        flock($fp, LOCK_UN);
    } else {
        $data = $default; // Could not lock
    }

    fclose($fp);
    return $data ?: $default;
}

function safe_write_json($filepath, $data)
{
    $fp = fopen($filepath, 'c+'); // Open for reading and writing
    if (!$fp)
        return false;

    // Exclusive Lock (Writer)
    if (flock($fp, LOCK_EX)) {
        ftruncate($fp, 0); // Clear file
        rewind($fp);
        fwrite($fp, json_encode($data, JSON_PRETTY_PRINT));
        fflush($fp); // Flush output before releasing lock
        flock($fp, LOCK_UN);
        $result = true;
    } else {
        $result = false;
    }

    fclose($fp);
    return $result;
}
?>