<?php
// ai/risk_guard.php
// LEVEL 5: EMOTION & RISK GUARD BRAIN
// Prevents overtrading, revenge trading, and huge drawdowns.

function check_risk_status($symbol)
{
    $file = __DIR__ . '/../trade_history.json';
    if (!file_exists($file))
        return ["status" => "OK", "msg" => ""];

    $history = json_decode(file_get_contents($file), true);
    if (!$history)
        return ["status" => "OK", "msg" => ""];

    // 1. Check Consecutive Losses (Revenge Guard)
    $losses = 0;
    $count = 0;
    // Scan backwards
    foreach (array_reverse($history) as $t) {
        if ($t['status'] === 'LOSS') {
            $losses++;
        } elseif ($t['status'] === 'WIN') {
            break; // Streak broken
        }
        $count++;
        if ($count > 5)
            break;
    }

    if ($losses >= 2) {
        // If last trade was very recent (< 30 mins)
        $last_trade = end($history);
        $last_time = $last_trade['timestamp'] ?? 0;
        if (time() - $last_time < 1800) { // 30 mins
            return ["status" => "COOLDOWN", "msg" => "🛑 Anti-Revenge: 2 Consecutive Losses. Cooldown active."];
        }
    }

    // 2. Check Daily Drawdown (Account Protection)
    // Assume we track daily PL. For now, simple recent trade check
    $today_pl = 0;
    $today_trades = 0;
    $date_today = date("Y-m-d");

    foreach ($history as $t) {
        if (strpos($t['date'], $date_today) === 0) {
            $today_pl += $t['pnl'] ?? 0;
            $today_trades++;
        }
    }

    // Hardcode simplistic protection: If -5000 loss (example) or -5%
    // Need user capital to know %. Assuming 50k capital = -2500
    // We will just use a generic "Too many losses today" if we don't know capital

    // 3. Overtrading Check
    if ($today_trades > 10) {
        return ["status" => "STOP_DAY", "msg" => "🛑 Overtrading: 10 trades today. Stop for the day."];
    }

    // --- SYSTEM GUARD UPGRADES (Survival Layer) ---

    // 4. Volatility Spike Check (ATR vs Moving Average of ATR)
    // We need candles to check this. If not passed, we skip or require them.
    // For safety, if we can't check, we proceed with caution (or pass candles to this func).
    // Assuming candles handled in engine, but for risk_guard isolation, we'll rely on Market State if available,
    // OR we can implement a lightweight check if data is passed.
    // *Modification*: We will accept optional $context data.

    // 5. News Shock Check (Sentiment Extreme)
    // If sentiment is too extreme (-0.8 or +0.8), it might be a news spike event (Gambling zone).
    if (isset($GLOBALS['latest_sentiment'])) {
        $s = $GLOBALS['latest_sentiment'];
        if (abs($s) > 0.8) {
            return ["status" => "NEWS_SPIKE", "msg" => "⚠️ News Shock Event (Sentiment " . round($s, 1) . "). High Risk."];
        }
    }

    // 6. Spread Health (Simulated for this version)
    // In real market, check Bid-Ask. Here, we check for 'Doji' chains or micro-candles which indicate low liquidity.
    // (Skipped for now as we don't have direct L1 data)

    return ["status" => "OK", "msg" => "Risk Guard Active"];
}

function check_volatility_safeguard($candles)
{
    if (count($candles) < 20)
        return true;

    $highs = array_column($candles, 'h');
    $lows = array_column($candles, 'l');
    $closes = array_column($candles, 'c');

    // Check last candle range vs avg range
    $last_range = end($highs) - end($lows);
    $avg_range = 0;
    for ($i = 1; $i <= 10; $i++) {
        $idx = count($candles) - 1 - $i;
        $avg_range += ($highs[$idx] - $lows[$idx]);
    }
    $avg_range /= 10;

    // If last candle is 3x larger than average -> Flash Crash/Spike
    if ($avg_range > 0 && $last_range > ($avg_range * 3)) {
        return ["status" => "VOLATILE_SPIKE", "msg" => "⚠️ Volatility Explosion (3x Range). Waiting to settle."];
    }

    return ["status" => "OK"];
}
?>