<?php
// ai/timeframe_confirm.php
// STAGE 2: DECISION ACCURACY
// Checks Higher Timeframe (HTF) Trend to confirm Lower Timeframe (LTF) Entry.

function check_timeframe_alignment($candles)
{
    $count = count($candles);
    if ($count < 60) {
        // Not enough data to simulate HTF confidently
        return ["aligned" => true, "trend" => "Unknown", "reason" => "Insufficient Data for HTF"];
    }

    // SIMULATE 1 HOUR CANDLES FROM 5 MIN CANDLES
    // 1 Hour = 12 x 5min candles.
    // We need at least 20 HTF candles to calculate EMA20.
    // That means 20 * 12 = 240 candles.
    // If we don't have that many, we fallback to 15m (3 candles).

    $aggregation = 12; // Default 1H
    if ($count < 240)
        $aggregation = 3; // Fallback to 15m

    $htf_candles = [];
    $chunk = [];

    foreach ($candles as $c) {
        $chunk[] = $c;
        if (count($chunk) == $aggregation) {
            // Aggregate
            $open = $chunk[0]['o'];
            $close = $chunk[$aggregation - 1]['c'];
            $high = max(array_column($chunk, 'h'));
            $low = min(array_column($chunk, 'l'));
            $vol = array_sum(array_column($chunk, 'v'));

            $htf_candles[] = ['o' => $open, 'h' => $high, 'l' => $low, 'c' => $close, 'v' => $vol];
            $chunk = [];
        }
    }

    if (count($htf_candles) < 10) {
        return ["aligned" => true, "trend" => "Neutral", "reason" => "Insufficient HTF Data"];
    }

    // CALCULATE HTF EMA
    $closes = array_column($htf_candles, 'c');
    $ema20 = calculate_ema_dynamic_htf($closes, 20); // Reuse logic or simpler avg
    $last_price = end($closes);

    $trend = "NEUTRAL";
    if ($last_price > $ema20)
        $trend = "BULLISH";
    if ($last_price < $ema20)
        $trend = "BEARISH";

    return [
        "aligned" => false, // Will be checked against signal
        "trend" => $trend,
        "ema" => $ema20,
        "reason" => "HTF ($aggregation x 5m) is $trend"
    ];
}


function calculate_ema_dynamic_htf($data, $period)
{
    if (count($data) <= $period)
        return array_sum($data) / count($data);
    $k = 2 / ($period + 1);
    $ema = $data[0];
    foreach ($data as $price) {
        $ema = ($price * $k) + ($ema * (1 - $k));
    }
    return $ema;
}
?>