<?php
// chat_engine.php - The Robust Brain of AI Guru
// 1. Suppress Display of Errors (Logs only)
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// 2. Buffer Output (To prevent stray characters breaking JSON)
ob_start();

header('Content-Type: application/json');

// 3. Shutdown Function for Fatal Errors
register_shutdown_function(function () {
    $error = error_get_last();
    if ($error && ($error['type'] === E_ERROR || $error['type'] === E_PARSE)) {
        ob_clean();
        echo json_encode(["reply" => "System busy (Code: " . $error['line'] . "). Please try again."]);
        exit;
    }
});

try {
    // 4. Input Parsing
    $input_raw = file_get_contents('php://input');
    $input = json_decode($input_raw, true);

    // DEFAULTS
    $msg = strtolower($input['message'] ?? '');
    $ctx = $input['context'] ?? [];

    $symbol = strtoupper($ctx['symbol'] ?? 'NIFTY');
    $price = 0;
    $signal = 'WAIT';
    $trend = 'Neutral';

    // 5. DATA FETCHING (Self-Reliant)
    if (file_exists('trade_engine.php')) {
        require_once 'trade_engine.php';

        // Always try to fetch fresh data if it's a specific stock query or explicitly requested
        if ($symbol !== 'NIFTY' && function_exists('run_analysis')) {
            $data = run_analysis($symbol, '1d'); // Use daily context for chat
            if ($data && isset($data['success']) && $data['success']) {
                $price = $data['price'];
                $analysis = $data['analysis']['intraday'] ?? null;
                if ($analysis) {
                    $signal = $analysis['signal'];
                    $trend = $analysis['details']['trend'] ?? 'Neutral';
                }
            } else {
                // Fallback / Offline
                $signal = 'WAIT';
                $trend = 'Uncertain';
            }
        }
    }

    // 6. INTENT CLASSIFICATION
    $reply = "";

    // Keywords (Malayalam / Tanglish / English)
    $is_buy = preg_match('/buy|vangamo|entry|long|bull/', $msg);
    $is_sell = preg_match('/sell|vilkamo|exit|short|bear/', $msg);
    $is_status = preg_match('/status|avasta|engane|noku|price|analysis/', $msg);
    $is_hello = preg_match('/hi|hello|namaskaram|mrng/', $msg);
    $is_help = preg_match('/help|enthu|what|aara|support/', $msg);

    // 7. ANSWER GENERATION
    if ($is_hello) {
        $reply = "Namaskaram! Njan ready aanu. <br> <b>$symbol</b>-ine kurichu enthu venamenkilum chodikkam.";
    } elseif ($is_help) {
        $reply = "🤖 Njan AI Guru aanu. <br> Enikku Stocks Analyze cheyyanum Buy/Sell signal tharanum kazhiyum.";
    } elseif ($is_buy) {
        if ($signal === 'BUY' || $signal === 'STRONG BUY') {
            $reply = "✅ **Buy Signal!** <br> $symbol strong aanu. Trend bullish aanu ($trend). <br> Stoploss vechu entry edukkam.";
        } elseif ($signal === 'SELL') {
            $reply = "⛔ **Risk aanu.** <br> Ippol Sell trend aanu. Buying safe alla.";
        } else {
            $reply = "⚠️ **Wait Cheyyu.** <br> Market clear alla. Volatility kooduthal aanu.";
        }
    } elseif ($is_sell) {
        if ($signal === 'SELL' || $signal === 'STRONG SELL') {
            $reply = "✅ **Exit/Short Signal.** <br> Trend weakness kanikkunnu. Profit book cheyyam.";
        } elseif ($signal === 'BUY') {
            $reply = "⛔ **Hold Cheyyuka.** <br> Trend iniyum positive aanu. Exit cheyyanda.";
        } else {
            $reply = "Partial profit book cheyyunnathu nallathanu. Safe aayi irikku.";
        }
    } elseif ($is_status || strpos($msg, $symbol) !== false || $msg === 'analysis') {
        $currColor = ($signal === 'BUY') ? '#0ecb81' : (($signal === 'SELL') ? '#f6465d' : '#eebb00');
        $reply = "📊 **$symbol Analysis:** <br> Status: <b style='color:$currColor'>$signal</b> <br> Trend: $trend <br> Price: $price <br> <br> <i>(Decision edukkumbol Stoploss marakkanda)</i>";
    } else {
        // Universal Default
        $reply = "Njan $symbol nokkunu... <br> Current Signal: <b>$signal</b> ($trend). <br> Thangalude plan enthanu? Buy aano Sell aano?";
    }

    // 8. OUTPUT
    ob_clean(); // Clean buffer before outputting JSON
    echo json_encode(["reply" => $reply]);

} catch (Exception $e) {
    ob_clean();
    // Simplified User Message, Log Real Error
    echo json_encode(["reply" => "Brain rebooting... Please ask again."]);
}
?>