<?php
// paper/api.php - Paper Trading Backend (V35 Unlimited)
header('Content-Type: application/json');
require_once '../core/session.php'; // This defines API_TOKEN
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$user_id = $_SESSION['user_id'] ?? 0;
$action = $_GET['action'] ?? '';
$token = $_GET['token'] ?? '';
$input = json_decode(file_get_contents('php://input'), true);

if ($token !== API_TOKEN && $_SERVER['HTTP_X_API_TOKEN'] !== API_TOKEN) {
    echo json_encode(['status' => 'error', 'msg' => 'Invalid API Token']);
    exit;
}

if (!$user_id) {
    echo json_encode(['status' => 'error', 'msg' => 'Unauthorized']);
    exit;
}

// DATABASE (JSON for simplicity, sharing logic with main app)
$db_file = __DIR__ . '/../users.json'; // Simulating DB
if (!file_exists($db_file))
    file_put_contents($db_file, json_encode([]));

function get_db()
{
    global $db_file;
    return json_decode(file_get_contents($db_file), true);
}
function save_db($data)
{
    global $db_file;
    file_put_contents($db_file, json_encode($data, JSON_PRETTY_PRINT));
}

$data = get_db();
if (!isset($data['users'][$user_id])) {
    $data['users'][$user_id] = ['balance' => 1000000, 'positions' => []]; // Default 10L
}
$user = &$data['users'][$user_id]; // Reference

// 1. GET PORTFOLIO
if ($action === 'get_portfolio') {
    echo json_encode(['status' => 'success', 'balance' => $user['balance'], 'positions' => array_values($user['positions'])]);
    exit;
}

// 2. SEARCH SYMBOL (Global Proxy)
if ($action === 'search_symbol') {
    $q = $_GET['query'] ?? '';
    // Use Yahoo Finance API Proxy (or internal logic)
    // Here we just return a mockup or forward request if implemented
    // For now, let's assume we use the main API's search if available, or fetch from Yahoo
    $url = "https://query1.finance.yahoo.com/v1/finance/search?q=" . urlencode($q);
    $json = file_get_contents($url);
    $res = json_decode($json, true);
    $out = [];
    if (isset($res['quotes'])) {
        foreach ($res['quotes'] as $r) {
            $out[] = ['symbol' => $r['symbol'], 'name' => $r['shortname'] ?? $r['longname'] ?? $r['symbol']];
        }
    }
    echo json_encode($out);
    exit;
}

// 3. ANALYZE (Search & add to list)
// This is handled by main api.php mostly, but if we need Paper-specific data:
// (Usually logic.js calls ../api.php?action=analyze, which is correct. This file handles TRADES)

// 4. RESET ACCOUNT
if ($action === 'reset_account') {
    $user['balance'] = 1000000000; // 100 Crore (Unlimited)
    $user['positions'] = [];
    save_db($data);
    echo json_encode(['status' => 'success', 'msg' => 'Account Reset to ₹100Cr']);
    exit;
}

// 5. PLACE ORDER
if ($action === 'place_order') {
    $symbol = $input['symbol'];
    $type = $input['type']; // BUY or SELL
    $qty = intval($input['qty']);
    $price = floatval($input['price']);

    if ($qty <= 0) {
        echo json_encode(['status' => 'error', 'msg' => 'Invalid Qty']);
        exit;
    }

    $cost = $price * $qty;
    if ($type === 'BUY') {
        if ($user['balance'] < $cost) {
            echo json_encode(['status' => 'error', 'msg' => 'Insufficient Funds']);
            exit;
        }
        $user['balance'] -= $cost;
    } else {
        // SELL (Short Selling allowed in Paper)
        $user['balance'] += $cost; // Credit immediately (Simplified margin)
    }

    $id = uniqid();
    $user['positions'][] = [
        'id' => $id,
        'symbol' => $symbol,
        'side' => $type,
        'qty' => $qty,
        'entry_price' => $price,
        'time' => time()
    ];
    save_db($data);
    echo json_encode(['status' => 'success', 'msg' => 'Order Executed', 'id' => $id]);
    exit;
}

// 6. CLOSE POSITION
if ($action === 'close_position') {
    $id = $input['id'];
    $exit_price = floatval($input['exit_price']);

    $found = false;
    foreach ($user['positions'] as $k => $pos) {
        if ($pos['id'] === $id) {
            $found = true;
            $qty = $pos['qty'];
            $entry = $pos['entry_price'];
            $side = $pos['side'];

            // P&L Calc
            $pnl = 0;
            if ($side === 'BUY') {
                $pnl = ($exit_price - $entry) * $qty;
                $user['balance'] += ($entry * $qty) + $pnl; // Return initial cost + profit
            } else {
                $pnl = ($entry - $exit_price) * $qty;
                $user['balance'] -= ($entry * $qty); // Remove the credit we gave earlier
                $user['balance'] += ($entry * $qty) + $pnl; // Add back adjusted (Simplified: just adjust PnL)
                // actually for SELL: we credited entry*qty. Now we debit exit*qty. 
                // balance = balance_before + entry*qty (at open).
                // close: we buy back at exit*qty. So balance -= exit*qty.
                // Net change = entry*qty - exit*qty = PnL.
                // So we need to subtract the exit cost.
                $user['balance'] -= ($exit_price * $qty);
                // Wait, logic above: "user['balance'] += $cost" on SELL.
                // So balance includes the credit.
                // Now we pay exit_price * qty.
                // user['balance'] -= ($exit_price * $qty). Correct.
            }

            unset($user['positions'][$k]);
            $user['positions'] = array_values($user['positions']);
            break;
        }
    }

    if ($found) {
        save_db($data);
        echo json_encode(['status' => 'success', 'msg' => 'Position Closed']);
    } else {
        echo json_encode(['status' => 'error', 'msg' => 'Position Not Found']);
    }
    exit;
}
?>