<?php
// paper/debug_tool.php - Universal Debugger
error_reporting(E_ALL);
ini_set('display_errors', 1);
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once '../core/session.php'; // Load API_TOKEN constant

$api_token = defined('API_TOKEN') ? API_TOKEN : ($_SESSION['api_token'] ?? 'UNKNOWN');
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Paper Trading Debugger</title>
    <style>
        body {
            background: #111;
            color: #eee;
            font-family: monospace;
            padding: 20px;
            line-height: 1.5;
        }

        .box {
            border: 1px solid #333;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            background: #1b2029;
        }

        .pass {
            color: #0ecb81;
            font-weight: bold;
        }

        .fail {
            color: #f6465d;
            font-weight: bold;
        }

        .warn {
            color: #eebb00;
        }

        button {
            background: #3b82f6;
            color: #fff;
            border: none;
            padding: 5px 10px;
            cursor: pointer;
            border-radius: 3px;
        }

        code {
            background: #222;
            padding: 2px 5px;
            border-radius: 3px;
        }
    </style>
</head>

<body>
    <h1>🛠️ Paper Trading Debugger (v2.0)</h1>

    <!-- 1. SERVER SIDE CHECKS -->
    <div class="box">
        <h3>1. Server-Side Check (PHP)</h3>
        <?php
        echo "<div>PHP Version: " . phpversion() . "</div>";
        echo "<div>Session Status: " . (session_status() === PHP_SESSION_ACTIVE ? "<span class='pass'>ACTIVE</span>" : "<span class='fail'>INACTIVE</span>") . "</div>";
        echo "<div>User ID: " . ($_SESSION['user_id'] ?? "<span class='fail'>MISSING</span>") . "</div>";
        echo "<div>API Token (Session): " . (defined('API_TOKEN') ? "<span class='pass'>PRESENT (Constant)</span>" : (isset($_SESSION['api_token']) ? "<span class='pass'>PRESENT (Session)</span>" : "<span class='fail'>MISSING</span>")) . "</div>";

        $files = ['index.php', 'logic.js', 'style.css', '../api.php', '../core/session.php'];
        foreach ($files as $f) {
            $path = __DIR__ . '/' . $f;
            $exists = file_exists($path);
            echo "<div>File <code>$f</code>: " . ($exists ? "<span class='pass'>FOUND</span> (" . filesize($path) . " bytes)" : "<span class='fail'>MISSING</span>") . "</div>";
        }
        ?>
    </div>

    <!-- 2. CLIENT SIDE CHECKS -->
    <div class="box">
        <h3>2. Client-Side Check (JS)</h3>
        <div id="jsLog">Waiting for JS...</div>
        <button onclick="runTests()">RUN CLIENT TESTS</button>
    </div>

    <script>
        const API_TOKEN = "<?php echo $api_token; ?>";
        const log = document.getElementById('jsLog');

        function print(msg, type = 'neutral') {
            const color = type === 'pass' ? '#0ecb81' : (type === 'fail' ? '#f6465d' : '#ccc');
            log.innerHTML += `<div style="color:${color}; margin-top:5px;">${msg}</div>`;
        }

        async function runTests() {
            log.innerHTML = "Starting Tests...<br>";

            // 1. Token Check
            if (API_TOKEN && API_TOKEN !== 'UNKNOWN') print("1. API Token Injected: PASS", 'pass');
            else print("1. API Token Injected: FAIL", 'fail');

            // 2. Fetch Check
            print("2. Testing API Connectivity (RELIANCE.NS)...");
            try {
                const start = Date.now();
                const res = await fetch(`../api.php?action=analyze&symbol=RELIANCE.NS&interval=1d&token=${API_TOKEN}`);
                const diff = Date.now() - start;

                if (res.status === 200) {
                    print(`   - HTTP 200 OK (${diff}ms)`, 'pass');
                    const text = await res.text();
                    try {
                        const json = JSON.parse(text);
                        if (json.price) print(`   - JSON Parsed. Price: ${json.price}`, 'pass');
                        else print(`   - JSON Invalid Structure: ${JSON.stringify(json).substr(0, 50)}`, 'fail');
                    } catch (e) {
                        print(`   - JSON Parse Error: ${e.message}. Raw: ${text.substr(0, 50)}`, 'fail');
                    }
                } else {
                    print(`   - HTTP Error: ${res.status}`, 'fail');
                }
            } catch (e) {
                print(`   - Network Error: ${e.message}`, 'fail');
            }

            // 3. Logic.js Check
            print("3. Checking logic.js integrity...");
            try {
                const scriptRes = await fetch('logic.js?v=' + Date.now());
                if (scriptRes.status === 200) {
                    const jsText = await scriptRes.text();
                    print(`   - logic.js loaded (${jsText.length} bytes)`, 'pass');
                    // Simple syntax check (very basic)
                    try {
                        new Function(jsText);
                        print("   - JS Syntax Check: PASS (No Parse Errors)", 'pass');
                    } catch (e) {
                        print(`   - JS Syntax Check: FAIL - ${e.message}`, 'fail');
                    }
                } else {
                    print("   - logic.js not found (404)", 'fail');
                }
            } catch (e) {
                print(`   - Error loading logic.js: ${e.message}`, 'fail');
            }
        }

        // Auto Run
        setTimeout(runTests, 1000);
    </script>
</body>

</html>